<?php

namespace MbeGroup\Author\Filament\Resources;

use MbeGroup\Author\Models\Author;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use MbeGroup\Author\Filament\Resources\AuthorResource\Pages\ListAuthors;
use MbeGroup\Author\Filament\Resources\AuthorResource\Pages\CreateAuthor;
use MbeGroup\Author\Filament\Resources\AuthorResource\Pages\EditAuthor;
use Filament\Actions;
use Filament\Forms;
use App\ValueObject\Status;

class AuthorResource extends Resource
{
    protected static ?string $model = Author::class;

    /**
     * Configure Status field with common hydration and dehydration logic
     */
    private static function configureStatusField(Forms\Components\Select $field): Forms\Components\Select
    {
        return $field
            ->afterStateHydrated(function (Forms\Components\Select $component, $state): void {
                if ($state instanceof Status) {
                    $component->state($state->value);
                }
            })
            ->dehydrateStateUsing(function ($state): int {
                return $state instanceof Status ? $state->value: (int) $state;
            })
            ->rule('integer');
    }

    public static function getNavigationGroup(): string
    {
        return __('author::filament.navigation.group.label');
    }

    public static function getNavigationLabel(): string
    {
        return __('author::filament.resources.author.navigation_label');
    }

    public static function getModelLabel(): string
    {
        return __('author::filament.resources.author.model_label');
    }

    public static function getPluralModelLabel(): string
    {
        return __('author::filament.resources.author.plural_model_label');
    }

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->schema([
                Forms\Components\TextInput::make('name')
                    ->label(__('author::filament.resources.author.form.fields.name'))
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (string $operation, $state, callable $set) {
                        if ($operation !== 'create') {
                            return;
                        }
                        $set('slug', \Illuminate\Support\Str::slug($state));
                    }),
            ])->columns(1);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label('ID')
                    ->sortable(),
                Tables\Columns\TextColumn::make('name')
                    ->label(__('author::filament.resources.author.table.columns.name'))
                    ->searchable()
            ])
            ->filters([
                          //
            ])
            ->recordActions([
                Actions\EditAction::make(),
                Actions\DeleteAction::make(),
            ])
            ->toolbarActions([
                Actions\BulkActionGroup::make([
                    Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->description('Version: ' . \Composer\InstalledVersions::getVersion('mbegroup/author'));
    }

    public static function getRelations(): array
    {
        return [

        ];
    }

    public static function getPages(): array
    {
        return [
            'index'  => ListAuthors::route('/'),
            'create' => CreateAuthor::route('/create'),
            'edit'   => EditAuthor::route('/{record}/edit'),
        ];
    }
}
