<?php

namespace Tests\Unit;

use MbeGroup\Author\Filament\AuthorPlugin;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

class AuthorPluginTest extends TestCase
{
    /**
     * Test sprawdzający czy AuthorPlugin można zinstancjonować
     */
    public function test_can_instantiate_plugin(): void
    {
        $plugin = new AuthorPlugin();
        
        $this->assertInstanceOf(AuthorPlugin::class, $plugin);
    }

    /**
     * Test sprawdzający czy klasa implementuje Plugin interface
     */
    public function test_implements_plugin_interface(): void
    {
        $reflection = new ReflectionClass(AuthorPlugin::class);
        
        // Sprawdzenie czy implementuje Plugin interface
        $this->assertTrue($reflection->implementsInterface(\Filament\Contracts\Plugin::class));
    }

    /**
     * Test sprawdzający czy klasa ma wymagane metody z Plugin interface
     */
    public function test_has_required_plugin_methods(): void
    {
        $reflection = new ReflectionClass(AuthorPlugin::class);
        
        // Sprawdzenie czy ma metodę getId
        $this->assertTrue($reflection->hasMethod('getId'), 'Plugin powinien mieć metodę getId');
        
        // Sprawdzenie czy ma metodę register
        $this->assertTrue($reflection->hasMethod('register'), 'Plugin powinien mieć metodę register');
        
        // Sprawdzenie czy ma metodę boot
        $this->assertTrue($reflection->hasMethod('boot'), 'Plugin powinien mieć metodę boot');
    }

    /**
     * Test sprawdzający czy getId zwraca poprawny identyfikator
     */
    public function test_get_id_returns_correct_identifier(): void
    {
        $plugin = new AuthorPlugin();
        
        $this->assertEquals('author', $plugin->getId());
    }

    /**
     * Test sprawdzający namespace klasy
     */
    public function test_class_namespace(): void
    {
        $reflection = new ReflectionClass(AuthorPlugin::class);
        
        $this->assertEquals('MbeGroup\Author\Filament', $reflection->getNamespaceName());
        $this->assertEquals('AuthorPlugin', $reflection->getShortName());
    }

    /**
     * Test sprawdzający czy metody są publiczne
     */
    public function test_plugin_methods_are_public(): void
    {
        $reflection = new ReflectionClass(AuthorPlugin::class);
        
        $getIdMethod = $reflection->getMethod('getId');
        $this->assertTrue($getIdMethod->isPublic(), 'Metoda getId powinna być publiczna');
        
        $registerMethod = $reflection->getMethod('register');
        $this->assertTrue($registerMethod->isPublic(), 'Metoda register powinna być publiczna');
        
        $bootMethod = $reflection->getMethod('boot');
        $this->assertTrue($bootMethod->isPublic(), 'Metoda boot powinna być publiczna');
    }

    /**
     * Test sprawdzający parametry metody register
     */
    public function test_register_method_parameters(): void
    {
        $reflection = new ReflectionClass(AuthorPlugin::class);
        $registerMethod = $reflection->getMethod('register');
        
        $parameters = $registerMethod->getParameters();
        $this->assertCount(1, $parameters, 'Metoda register powinna mieć dokładnie jeden parametr');
        
        $panelParam = $parameters[0];
        $this->assertEquals('panel', $panelParam->getName());
    }

    /**
     * Test sprawdzający parametry metody boot
     */
    public function test_boot_method_parameters(): void
    {
        $reflection = new ReflectionClass(AuthorPlugin::class);
        $bootMethod = $reflection->getMethod('boot');
        
        $parameters = $bootMethod->getParameters();
        $this->assertCount(1, $parameters, 'Metoda boot powinna mieć dokładnie jeden parametr');
        
        $panelParam = $parameters[0];
        $this->assertEquals('panel', $panelParam->getName());
    }

    /**
     * Test sprawdzający czy nie ma niepotrzebnych właściwości publicznych
     */
    public function test_no_unexpected_public_properties(): void
    {
        $reflection = new ReflectionClass(AuthorPlugin::class);
        $publicProperties = [];
        
        foreach ($reflection->getProperties(\ReflectionProperty::IS_PUBLIC) as $property) {
            if ($property->class === AuthorPlugin::class) {
                $publicProperties[] = $property->name;
            }
        }
        
        // Plugin nie powinien mieć żadnych publicznych właściwości
        $this->assertEmpty($publicProperties, 'Plugin nie powinien mieć publicznych właściwości');
    }
}