<?php

namespace Tests\Unit;

use MbeGroup\Author\Filament\Resources\AuthorResource;
use PHPUnit\Framework\TestCase;
use ReflectionClass;
use Filament\Schemas\Schema;
use Filament\Tables\Table;

class AuthorResourceFormTableTest extends TestCase
{
    /**
     * Test sprawdzający czy metoda form ma odpowiednie właściwości
     */
    public function test_form_method_returns_schema(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        $this->assertTrue($reflection->hasMethod('form'), 'Resource powinien mieć metodę form');

        $formMethod = $reflection->getMethod('form');
        $this->assertTrue($formMethod->isStatic(), 'Metoda form powinna być statyczna');
        $this->assertTrue($formMethod->isPublic(), 'Metoda form powinna być publiczna');

        $returnType = $formMethod->getReturnType();
        $this->assertNotNull($returnType, 'Metoda form powinna mieć return type');
        $this->assertEquals('Filament\Schemas\Schema', $returnType->getName(), 'Metoda form powinna zwracać Schema');
    }

    /**
     * Test sprawdzający czy metoda table ma odpowiednie właściwości
     */
    public function test_table_method_returns_table(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        $this->assertTrue($reflection->hasMethod('table'), 'Resource powinien mieć metodę table');

        $tableMethod = $reflection->getMethod('table');
        $this->assertTrue($tableMethod->isStatic(), 'Metoda table powinna być statyczna');
        $this->assertTrue($tableMethod->isPublic(), 'Metoda table powinna być publiczna');

        $returnType = $tableMethod->getReturnType();
        $this->assertNotNull($returnType, 'Metoda table powinna mieć return type');
        $this->assertEquals('Filament\Tables\Table', $returnType->getName(), 'Metoda table powinna zwracać Table');
    }

    /**
     * Test sprawdzający metodę configureStatusField
     */
    public function test_configure_status_field_method(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        $this->assertTrue($reflection->hasMethod('configureStatusField'));

        $method = $reflection->getMethod('configureStatusField');
        $this->assertTrue($method->isStatic());
        $this->assertTrue($method->isPrivate());
    }

    /**
     * Test sprawdzający parametry metody configureStatusField
     */
    public function test_configure_status_field_parameters(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);
        $method = $reflection->getMethod('configureStatusField');

        $parameters = $method->getParameters();
        $this->assertCount(1, $parameters);

        $fieldParam = $parameters[0];
        $this->assertEquals('field', $fieldParam->getName());
    }

    /**
     * Test sprawdzający czy form schema ma odpowiednią strukturę
     */
    public function test_form_schema_structure(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);
        $formMethod = $reflection->getMethod('form');

        // Sprawdzenie czy metoda istnieje i ma odpowiednie parametry
        $this->assertTrue($formMethod->isStatic());
        $this->assertTrue($formMethod->isPublic());

        $parameters = $formMethod->getParameters();
        $this->assertCount(1, $parameters);
        $this->assertEquals('schema', $parameters[0]->getName());
    }

    /**
     * Test sprawdzający czy table ma odpowiednią strukturę
     */
    public function test_table_structure(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);
        $tableMethod = $reflection->getMethod('table');

        // Sprawdzenie czy metoda istnieje i ma odpowiednie parametry
        $this->assertTrue($tableMethod->isStatic());
        $this->assertTrue($tableMethod->isPublic());

        $parameters = $tableMethod->getParameters();
        $this->assertCount(1, $parameters);
        $this->assertEquals('table', $parameters[0]->getName());
    }

    /**
     * Test sprawdzający zwracany typ metod form i table
     */
    public function test_form_and_table_return_types(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        $formMethod = $reflection->getMethod('form');
        $formReturnType = $formMethod->getReturnType();
        $this->assertNotNull($formReturnType);
        $this->assertEquals('Filament\Schemas\Schema', $formReturnType->getName());

        $tableMethod = $reflection->getMethod('table');
        $tableReturnType = $tableMethod->getReturnType();
        $this->assertNotNull($tableReturnType);
        $this->assertEquals('Filament\Tables\Table', $tableReturnType->getName());
    }

    /**
     * Test sprawdzający czy metody form i table nie mają dodatkowych zależności
     */
    public function test_form_and_table_dependencies(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        // Test czy metoda form nie wymaga dodatkowych zależności poza schematem
        $formMethod = $reflection->getMethod('form');
        $this->assertCount(1, $formMethod->getParameters());

        // Test czy metoda table nie wymaga dodatkowych zależności poza table
        $tableMethod = $reflection->getMethod('table');
        $this->assertCount(1, $tableMethod->getParameters());
    }

    /**
     * Test sprawdzający czy metody są poprawnie zadeklarowane jako publiczne i statyczne
     */
    public function test_method_visibility_and_scope(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        $methods = ['form', 'table', 'getNavigationGroup', 'getNavigationLabel', 'getModelLabel', 'getPluralModelLabel'];

        foreach ($methods as $methodName) {
            $method = $reflection->getMethod($methodName);
            $this->assertTrue($method->isPublic(), "Metoda {$methodName} powinna być publiczna");
            $this->assertTrue($method->isStatic(), "Metoda {$methodName} powinna być statyczna");
        }
    }

    /**
     * Test sprawdzający czy AuthorResource ma wszystkie wymagane metody
     */
    public function test_resource_has_all_required_methods(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        $requiredMethods = [
            'form',
            'table',
            'getPages',
            'getRelations',
            'getNavigationGroup',
            'getNavigationLabel',
            'getModelLabel',
            'getPluralModelLabel'
        ];

        foreach ($requiredMethods as $method) {
            $this->assertTrue(
                $reflection->hasMethod($method),
                "AuthorResource powinien mieć metodę {$method}"
            );
        }
    }

    /**
     * Test sprawdzający czy configureStatusField zwraca odpowiedni typ
     */
    public function test_configure_status_field_return_type(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);
        $method = $reflection->getMethod('configureStatusField');

        $returnType = $method->getReturnType();
        $this->assertNotNull($returnType);
        $this->assertEquals('Filament\Forms\Components\Select', $returnType->getName());
    }

    /**
     * Test sprawdzający czy wszystkie metody nawigacji zwracają string
     */
    public function test_navigation_methods_return_string(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        $navigationMethods = [
            'getNavigationGroup',
            'getNavigationLabel',
            'getModelLabel',
            'getPluralModelLabel'
        ];

        foreach ($navigationMethods as $methodName) {
            $method = $reflection->getMethod($methodName);
            $returnType = $method->getReturnType();

            $this->assertNotNull($returnType, "Metoda {$methodName} powinna mieć zdefiniowany return type");
            $this->assertEquals('string', $returnType->getName(), "Metoda {$methodName} powinna zwracać string");
        }
    }
}
