<?php

namespace Tests\Unit;

use MbeGroup\Author\Filament\Resources\AuthorResource;
use MbeGroup\Author\Models\Author;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

class AuthorResourceTest extends TestCase
{
    /**
     * Test sprawdzający czy AuthorResource można zinstancjonować
     */
    public function test_can_instantiate_resource(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        $this->assertInstanceOf(ReflectionClass::class, $reflection);
        $this->assertEquals('AuthorResource', $reflection->getShortName());
    }

    /**
     * Test sprawdzający czy klasa dziedziczy po Resource
     */
    public function test_extends_resource(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        // Sprawdzenie czy dziedziczy po Filament Resource
        $this->assertTrue($reflection->isSubclassOf(\Filament\Resources\Resource::class));
    }

    /**
     * Test sprawdzający model class
     */
    public function test_model_configuration(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);
        $modelProperty = $reflection->getProperty('model');
        $modelProperty->setAccessible(true);

        $this->assertEquals(Author::class, $modelProperty->getValue());
    }

    /**
     * Test sprawdzający namespace klasy
     */
    public function test_class_namespace(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        $this->assertEquals('MbeGroup\Author\Filament\Resources', $reflection->getNamespaceName());
        $this->assertEquals('AuthorResource', $reflection->getShortName());
    }

    /**
     * Test sprawdzający czy ma wymagane metody statyczne
     */
    public function test_has_required_static_methods(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        // Sprawdzenie czy ma metodę form
        $this->assertTrue($reflection->hasMethod('form'), 'Resource powinien mieć metodę form');

        // Sprawdzenie czy ma metodę table
        $this->assertTrue($reflection->hasMethod('table'), 'Resource powinien mieć metodę table');

        // Sprawdzenie czy ma metodę getPages
        $this->assertTrue($reflection->hasMethod('getPages'), 'Resource powinien mieć metodę getPages');

        // Sprawdzenie czy ma metodę getRelations
        $this->assertTrue($reflection->hasMethod('getRelations'), 'Resource powinien mieć metodę getRelations');
    }

    /**
     * Test sprawdzający metody nawigacji
     */
    public function test_has_navigation_methods(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        // Sprawdzenie czy ma metodę getNavigationGroup
        $this->assertTrue($reflection->hasMethod('getNavigationGroup'), 'Resource powinien mieć metodę getNavigationGroup');

        // Sprawdzenie czy ma metodę getNavigationLabel
        $this->assertTrue($reflection->hasMethod('getNavigationLabel'), 'Resource powinien mieć metodę getNavigationLabel');

        // Sprawdzenie czy ma metodę getModelLabel
        $this->assertTrue($reflection->hasMethod('getModelLabel'), 'Resource powinien mieć metodę getModelLabel');

        // Sprawdzenie czy ma metodę getPluralModelLabel
        $this->assertTrue($reflection->hasMethod('getPluralModelLabel'), 'Resource powinien mieć metodę getPluralModelLabel');
    }

    /**
     * Test sprawdzający czy metody nawigacji są statyczne
     */
    public function test_navigation_methods_are_static(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        $getNavigationGroupMethod = $reflection->getMethod('getNavigationGroup');
        $this->assertTrue($getNavigationGroupMethod->isStatic(), 'Metoda getNavigationGroup powinna być statyczna');

        $getNavigationLabelMethod = $reflection->getMethod('getNavigationLabel');
        $this->assertTrue($getNavigationLabelMethod->isStatic(), 'Metoda getNavigationLabel powinna być statyczna');

        $getModelLabelMethod = $reflection->getMethod('getModelLabel');
        $this->assertTrue($getModelLabelMethod->isStatic(), 'Metoda getModelLabel powinna być statyczna');
    }

    /**
     * Test sprawdzający parametry metody form
     */
    public function test_form_method_parameters(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);
        $formMethod = $reflection->getMethod('form');

        $parameters = $formMethod->getParameters();
        $this->assertCount(1, $parameters, 'Metoda form powinna mieć dokładnie jeden parametr');

        $schemaParam = $parameters[0];
        $this->assertEquals('schema', $schemaParam->getName());
    }

    /**
     * Test sprawdzający parametry metody table
     */
    public function test_table_method_parameters(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);
        $tableMethod = $reflection->getMethod('table');

        $parameters = $tableMethod->getParameters();
        $this->assertCount(1, $parameters, 'Metoda table powinna mieć dokładnie jeden parametr');

        $tableParam = $parameters[0];
        $this->assertEquals('table', $tableParam->getName());
    }

    /**
     * Test sprawdzający czy metody form i table są statyczne
     */
    public function test_form_and_table_methods_are_static(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        $formMethod = $reflection->getMethod('form');
        $this->assertTrue($formMethod->isStatic(), 'Metoda form powinna być statyczna');

        $tableMethod = $reflection->getMethod('table');
        $this->assertTrue($tableMethod->isStatic(), 'Metoda table powinna być statyczna');
    }

    /**
     * Test sprawdzający czy getPages zwraca array
     */
    public function test_get_pages_returns_array(): void
    {
        $pages = AuthorResource::getPages();

        $this->assertIsArray($pages, 'Metoda getPages powinna zwracać array');
        $this->assertNotEmpty($pages, 'Metoda getPages nie powinna zwracać pustego array');
    }

    /**
     * Test sprawdzający czy getRelations zwraca array
     */
    public function test_get_relations_returns_array(): void
    {
        $relations = AuthorResource::getRelations();

        $this->assertIsArray($relations, 'Metoda getRelations powinna zwracać array');
    }

    /**
     * Test sprawdzający czy ma prywatną metodę configureStatusField
     */
    public function test_has_configure_status_field_method(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        $this->assertTrue($reflection->hasMethod('configureStatusField'), 'Resource powinien mieć metodę configureStatusField');

        $configureStatusFieldMethod = $reflection->getMethod('configureStatusField');
        $this->assertTrue($configureStatusFieldMethod->isStatic(), 'Metoda configureStatusField powinna być statyczna');
        $this->assertTrue($configureStatusFieldMethod->isPrivate(), 'Metoda configureStatusField powinna być prywatna');
    }

    /**
     * Test sprawdzający kluczowe strony w getPages
     */
    public function test_get_pages_has_required_pages(): void
    {
        $pages = AuthorResource::getPages();

        $this->assertArrayHasKey('index', $pages, 'Pages powinny zawierać stronę index');
        $this->assertArrayHasKey('create', $pages, 'Pages powinny zawierać stronę create');
        $this->assertArrayHasKey('edit', $pages, 'Pages powinny zawierać stronę edit');
    }

    /**
     * Test sprawdzający czy metody nawigacji mają odpowiednie właściwości
     */
    public function test_navigation_methods_return_strings(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        // Sprawdzenie czy metody nawigacyjne istnieją i są właściwe
        $navigationMethods = ['getNavigationGroup', 'getNavigationLabel', 'getModelLabel', 'getPluralModelLabel'];

        foreach ($navigationMethods as $methodName) {
            $this->assertTrue($reflection->hasMethod($methodName), "Metoda {$methodName} powinna istnieć");

            $method = $reflection->getMethod($methodName);
            $this->assertTrue($method->isStatic(), "Metoda {$methodName} powinna być statyczna");
            $this->assertTrue($method->isPublic(), "Metoda {$methodName} powinna być publiczna");

            $returnType = $method->getReturnType();
            $this->assertNotNull($returnType, "Metoda {$methodName} powinna mieć return type");
            $this->assertEquals('string', $returnType->getName(), "Metoda {$methodName} powinna zwracać string");
        }
    }
}
