<?php

namespace Tests\Unit;

use MbeGroup\Author\Filament\AuthorPlugin;
use MbeGroup\Author\Models\Author;
use MbeGroup\Author\Filament\Resources\AuthorResource;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

class AuthorIntegrationTest extends TestCase
{
    /**
     * Test sprawdzający całkowitą integrację komponentów Author
     */
    public function test_author_components_integration(): void
    {
        // Sprawdzenie czy wszystkie główne klasy istnieją
        $this->assertTrue(class_exists(Author::class), 'Klasa Author powinna istnieć');
        $this->assertTrue(class_exists(AuthorPlugin::class), 'Klasa AuthorPlugin powinna istnieć');
        $this->assertTrue(class_exists(AuthorResource::class), 'Klasa AuthorResource powinna istnieć');
    }

    /**
     * Test sprawdzający czy AuthorResource jest poprawnie skonfigurowany dla modelu Author
     */
    public function test_author_resource_model_configuration(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);
        $modelProperty = $reflection->getProperty('model');
        $modelProperty->setAccessible(true);

        $this->assertEquals(Author::class, $modelProperty->getValue(),
            'AuthorResource powinien być skonfigurowany dla modelu Author');
    }

    /**
     * Test sprawdzający namespace consistency
     */
    public function test_namespace_consistency(): void
    {
        $authorReflection = new ReflectionClass(Author::class);
        $pluginReflection = new ReflectionClass(AuthorPlugin::class);
        $resourceReflection = new ReflectionClass(AuthorResource::class);

        // Sprawdzenie czy wszystkie klasy są w przestrzeni nazw MbeGroup\Author
        $this->assertStringStartsWith('MbeGroup\Author', $authorReflection->getNamespaceName());
        $this->assertStringStartsWith('MbeGroup\Author', $pluginReflection->getNamespaceName());
        $this->assertStringStartsWith('MbeGroup\Author', $resourceReflection->getNamespaceName());
    }

    /**
     * Test sprawdzający dostępność podstawowych metod w różnych komponentach
     */
    public function test_component_methods_availability(): void
    {
        // Test AuthorPlugin
        $plugin = new AuthorPlugin();
        $this->assertEquals('author', $plugin->getId());

        // Test Author model fillable fields
        $author = new Author();
        $fillable = $author->getFillable();
        $requiredFields = ['name', 'email', 'position'];

        foreach ($requiredFields as $field) {
            $this->assertContains($field, $fillable, "Pole {$field} powinno być fillable");
        }
    }

    /**
     * Test sprawdzający czy AuthorResource ma odpowiednie strony
     */
    public function test_author_resource_pages_structure(): void
    {
        $pages = AuthorResource::getPages();

        $requiredPages = ['index', 'create', 'edit'];

        foreach ($requiredPages as $page) {
            $this->assertArrayHasKey($page, $pages, "Strona {$page} powinna być zdefiniowana");
        }
    }

    /**
     * Test sprawdzający czy można utworzyć instancję Author z danymi
     */
    public function test_author_model_instantiation_with_data(): void
    {
        $data = [
            'name' => 'Test Author',
            'email' => 'test@example.com',
            'position' => 'Test Position'
        ];

        $author = new Author($data);

        $this->assertEquals('Test Author', $author->name);
        $this->assertEquals('test@example.com', $author->email);
        $this->assertEquals('Test Position', $author->position);
    }

    /**
     * Test sprawdzający czy AuthorPlugin poprawnie rejestruje resource
     */
    public function test_plugin_registers_resource(): void
    {
        $plugin = new AuthorPlugin();
        $reflection = new ReflectionClass($plugin);

        // Sprawdzenie czy ma metodę register
        $this->assertTrue($reflection->hasMethod('register'), 'Plugin powinien mieć metodę register');

        // Sprawdzenie czy register jest publiczna
        $registerMethod = $reflection->getMethod('register');
        $this->assertTrue($registerMethod->isPublic(), 'Metoda register powinna być publiczna');
    }

    /**
     * Test sprawdzający zgodność interface Plugin
     */
    public function test_plugin_interface_compliance(): void
    {
        $plugin = new AuthorPlugin();
        $reflection = new ReflectionClass($plugin);

        // Sprawdzenie implementacji interface
        $this->assertTrue($reflection->implementsInterface(\Filament\Contracts\Plugin::class));

        // Sprawdzenie wymaganych metod
        $requiredMethods = ['getId', 'register', 'boot'];
        foreach ($requiredMethods as $method) {
            $this->assertTrue($reflection->hasMethod($method), "Plugin powinien mieć metodę {$method}");
        }
    }

    /**
     * Test sprawdzający czy AuthorResource ma poprawnie skonfigurowane metody nawigacji
     */
    public function test_resource_navigation_configuration(): void
    {
        $reflection = new ReflectionClass(AuthorResource::class);

        // Sprawdzenie czy metody nawigacyjne istnieją
        $this->assertTrue($reflection->hasMethod('getNavigationGroup'), 'Resource powinien mieć metodę getNavigationGroup');
        $this->assertTrue($reflection->hasMethod('getNavigationLabel'), 'Resource powinien mieć metodę getNavigationLabel');
        $this->assertTrue($reflection->hasMethod('getModelLabel'), 'Resource powinien mieć metodę getModelLabel');
        $this->assertTrue($reflection->hasMethod('getPluralModelLabel'), 'Resource powinien mieć metodę getPluralModelLabel');

        // Sprawdzenie czy metody są statyczne i publiczne
        $navigationMethods = ['getNavigationGroup', 'getNavigationLabel', 'getModelLabel', 'getPluralModelLabel'];
        foreach ($navigationMethods as $methodName) {
            $method = $reflection->getMethod($methodName);
            $this->assertTrue($method->isStatic(), "Metoda {$methodName} powinna być statyczna");
            $this->assertTrue($method->isPublic(), "Metoda {$methodName} powinna być publiczna");

            $returnType = $method->getReturnType();
            $this->assertNotNull($returnType, "Metoda {$methodName} powinna mieć return type");
            $this->assertEquals('string', $returnType->getName(), "Metoda {$methodName} powinna zwracać string");
        }
    }

    /**
     * Test sprawdzający czy model ma poprawną konfigurację MongoDB
     */
    public function test_author_model_mongodb_configuration(): void
    {
        $author = new Author();

        $this->assertEquals('mongodb', $author->getConnectionName(),
            'Author powinien używać połączenia mongodb');
        $this->assertEquals('authors', $author->getTable(),
            'Author powinien używać kolekcji authors');
    }

    /**
     * Test sprawdzający kompletność pakietu
     */
    public function test_package_completeness(): void
    {
        // Sprawdzenie czy wszystkie podstawowe komponenty są obecne
        $requiredClasses = [
            Author::class,
            AuthorPlugin::class,
            AuthorResource::class,
        ];

        foreach ($requiredClasses as $class) {
            $this->assertTrue(class_exists($class), "Klasa {$class} powinna istnieć");

            $reflection = new ReflectionClass($class);
            $this->assertFalse($reflection->isAbstract(), "Klasa {$class} nie powinna być abstrakcyjna");
            $this->assertTrue($reflection->isInstantiable(), "Klasa {$class} powinna być instancjonowalna");
        }
    }
}
