<?php

namespace Tests\Unit;

use MbeGroup\Author\Models\Author;
use PHPUnit\Framework\TestCase;
use ReflectionClass;
use App\ValueObject\Status;

class AuthorModelTest extends TestCase
{
    /**
     * Test sprawdzający czy model Author można zinstancjonować
     */
    public function test_can_instantiate_author_model(): void
    {
        $author = new Author();

        $this->assertInstanceOf(Author::class, $author);
    }

    /**
     * Test sprawdzający właściwości modelu
     */
    public function test_model_properties(): void
    {
        $reflection = new ReflectionClass(Author::class);

        // Sprawdzenie czy dziedziczy po odpowiedniej klasie
        $this->assertTrue($reflection->isSubclassOf(\MongoDB\Laravel\Eloquent\Model::class));
    }

    /**
     * Test sprawdzający konfigurację połączenia
     */
    public function test_model_connection_configuration(): void
    {
        $author = new Author();

        // Sprawdzenie konfiguracji połączenia MongoDB
        $this->assertEquals('mongodb', $author->getConnectionName());
    }

    /**
     * Test sprawdzający nazwę kolekcji
     */
    public function test_model_collection_name(): void
    {
        $author = new Author();

        // Sprawdzenie nazwy kolekcji
        $this->assertEquals('authors', $author->getTable());
    }

    /**
     * Test sprawdzający fillable fields
     */
    public function test_fillable_fields(): void
    {
        $author = new Author();
        $expectedFillable = [
            'name',
            'position',
            'company_name',
            'type_id',
            'email',
            'image',
            'biography',
            'status',
        ];

        $this->assertEquals($expectedFillable, $author->getFillable());
    }

    /**
     * Test sprawdzający castowanie
     */
    public function test_casts_configuration(): void
    {
        $author = new Author();
        $casts = $author->getCasts();

        $this->assertArrayHasKey('status', $casts);
        $this->assertEquals(Status::class, $casts['status']);
    }

    /**
     * Test sprawdzający czy można ustawić name
     */
    public function test_can_set_name(): void
    {
        $author = new Author();
        $author->name = 'Jan Kowalski';

        $this->assertEquals('Jan Kowalski', $author->name);
    }

    /**
     * Test sprawdzający czy można ustawić email
     */
    public function test_can_set_email(): void
    {
        $author = new Author();
        $author->email = 'jan.kowalski@example.com';

        $this->assertEquals('jan.kowalski@example.com', $author->email);
    }

    /**
     * Test sprawdzający czy można ustawić pozycję
     */
    public function test_can_set_position(): void
    {
        $author = new Author();
        $author->position = 'Senior Developer';

        $this->assertEquals('Senior Developer', $author->position);
    }

    /**
     * Test sprawdzający czy można ustawić nazwę firmy
     */
    public function test_can_set_company_name(): void
    {
        $author = new Author();
        $author->company_name = 'Example Corp';

        $this->assertEquals('Example Corp', $author->company_name);
    }

    /**
     * Test sprawdzający czy można ustawić biografię
     */
    public function test_can_set_biography(): void
    {
        $author = new Author();
        $author->biography = 'Doświadczony programista z wieloletnim stażem.';

        $this->assertEquals('Doświadczony programista z wieloletnim stażem.', $author->biography);
    }

    /**
     * Test sprawdzający czy model ma metodę type()
     */
    public function test_has_type_relation(): void
    {
        $reflection = new ReflectionClass(Author::class);

        $this->assertTrue($reflection->hasMethod('type'), 'Model powinien mieć metodę type()');
    }

    /**
     * Test sprawdzający czy można wypełnić wszystkie fillable fields jednocześnie
     */
    public function test_mass_assignment(): void
    {
        $data = [
            'name' => 'Anna Nowak',
            'position' => 'Frontend Developer',
            'company_name' => 'Tech Solutions',
            'type_id' => '12345',
            'email' => 'anna.nowak@tech.com',
            'image' => 'path/to/image.jpg',
            'biography' => 'Specjalistka od React i Vue.js',
            'status' => 1
        ];

        $author = new Author($data);

        // Sprawdzamy tylko podstawowe fields, pomijamy status który jest castowany
        $this->assertEquals('Anna Nowak', $author->name);
        $this->assertEquals('Frontend Developer', $author->position);
        $this->assertEquals('Tech Solutions', $author->company_name);
        $this->assertEquals('12345', $author->type_id);
        $this->assertEquals('anna.nowak@tech.com', $author->email);
        $this->assertEquals('path/to/image.jpg', $author->image);
        $this->assertEquals('Specjalistka od React i Vue.js', $author->biography);
        // Status jest castowany więc sprawdzamy tylko czy istnieje
        $this->assertNotNull($author->status);
    }

    /**
     * Test sprawdzający czy model nie pozwala na ustawienie nie-fillable fields
     */
    public function test_guards_against_non_fillable_fields(): void
    {
        $author = new Author();

        // Sprawdzenie czy id nie jest fillable
        $this->assertNotContains('id', $author->getFillable());

        // Sprawdzenie czy created_at nie jest fillable
        $this->assertNotContains('created_at', $author->getFillable());

        // Sprawdzenie czy updated_at nie jest fillable
        $this->assertNotContains('updated_at', $author->getFillable());
    }
}
