<?php

namespace MbeGroup\Author\Filament\Resources;

use MbeGroup\Author\Models\Author;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use MbeGroup\Author\Filament\Resources\AuthorResource\Pages\ListAuthors;
use MbeGroup\Author\Filament\Resources\AuthorResource\Pages\CreateAuthor;
use MbeGroup\Author\Filament\Resources\AuthorResource\Pages\EditAuthor;
use Filament\Actions;
use Filament\Forms;
use App\ValueObject\Status;
use App\Helpers\DictionaryHelper;
use App\Filament\Components\RichEditorFormComponents;

class AuthorResource extends Resource
{
    protected static ?string $model = Author::class;

      /**
     * Configure Status field with common hydration and dehydration logic
     */
    private static function configureStatusField(Forms\Components\Select $field): Forms\Components\Select
    {
        return $field
            ->afterStateHydrated(function (Forms\Components\Select $component, $state): void {
                if ($state instanceof Status) {
                    $component->state($state->value);
                }
            })
            ->dehydrateStateUsing(function ($state): int {
                return $state instanceof Status ? $state->value: (int) $state;
            })
            ->rule('integer');
    }

    public static function getNavigationGroup(): string
    {
        return __('author::filament.navigation.group.label');
    }

    public static function getNavigationLabel(): string
    {
        return __('author::filament.resources.author.navigation_label');
    }

    public static function getModelLabel(): string
    {
        return __('author::filament.resources.author.model_label');
    }

    public static function getPluralModelLabel(): string
    {
        return __('author::filament.resources.author.plural_model_label');
    }

    public static function form(Schema $schema): Schema
    {
        $dictionaryOptions = [
            'author_type' => DictionaryHelper::getHierarchicalNamesByType('author_type'),
        ];

        $statusOptions = Status::getLabels();

        return $schema
            ->schema([
                Forms\Components\TextInput::make('name')
                    ->label(__('author::filament.resources.author.form.fields.name'))
                    ->required()
                    ->maxLength(255)
                    ->live(onBlur: true)
                    ->afterStateUpdated(function (string $operation, $state, callable $set) {
                        if ($operation !== 'create') {
                            return;
                        }
                        $set('slug', \Illuminate\Support\Str::slug($state));
                    }),
                Forms\Components\TextInput::make('position')
                    ->label(__('author::filament.resources.author.form.fields.position'))
                    ->maxLength(255),
                Forms\Components\TextInput::make('company_name')
                    ->label(__('author::filament.resources.author.form.fields.company_name'))
                    ->maxLength(255),
                Forms\Components\Select::make('type_id')
                    ->label(__('author::filament.resources.author.form.fields.author_type'))
                    ->options($dictionaryOptions['author_type'])
                    ->searchable()
                    ->allowHtml()
                    ->required(),
                Forms\Components\TextInput::make('email')
                    ->label(__('author::filament.resources.author.form.fields.email'))
                    ->email()
                    ->maxLength(255),
                Forms\Components\FileUpload::make('image')
                    ->label(__('author::filament.resources.author.form.fields.image'))
                    ->image()
                    ->imageEditor()
                    ->imagePreviewHeight('150')
                    ->directory('authors')
                    ->disk('public')
                    ->downloadable()
                    ->openable()
                    ->maxSize(2048),
                RichEditorFormComponents::getRichEditor(
                    __('author::filament.resources.author.form.fields.biography'),
                    'biography',
                ),
                self::configureStatusField(
                    Forms\Components\Select::make('status')
                        ->label(__('author::filament.resources.author.form.fields.status'))
                        ->options($statusOptions)
                        ->required()
                        ->default(Status::INACTIVE)
                ),
            ])->columns(1);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label('ID')
                    ->sortable(),
                Tables\Columns\TextColumn::make('name')
                    ->label(__('author::filament.resources.author.table.columns.name'))
                    ->searchable(),
                Tables\Columns\TextColumn::make('position')
                    ->label(__('author::filament.resources.author.table.columns.position'))
                    ->searchable(),
                Tables\Columns\TextColumn::make('company_name')
                    ->label(__('author::filament.resources.author.table.columns.company_name'))
                    ->searchable(),
                Tables\Columns\TextColumn::make('type.name')
                    ->label(__('author::filament.resources.author.table.columns.type'))
                    ->searchable(),
                Tables\Columns\IconColumn::make('status')
                    ->label(__('author::filament.resources.author.table.columns.status'))
                    ->state(fn ($record) => Status::int($record->status))
                    ->icon(fn ($state) => Status::icon($state))
                    ->color(fn ($state) => Status::color($state))
                    ->tooltip(fn ($state) => Status::label($state))
                    ->sortable(),
            ])
            ->filters([
                            //
            ])
            ->recordActions([
                Actions\EditAction::make(),
                Actions\DeleteAction::make(),
            ])
            ->toolbarActions([
                Actions\BulkActionGroup::make([
                    Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->description('Version: ' . \Composer\InstalledVersions::getVersion('mbegroup/author'));
    }

    public static function getRelations(): array
    {
        return [

        ];
    }

    public static function getPages(): array
    {
        return [
            'index'  => ListAuthors::route('/'),
            'create' => CreateAuthor::route('/create'),
            'edit'   => EditAuthor::route('/{record}/edit'),
        ];
    }
}
