<?php

namespace Tests\Unit;

use MbeGroup\Author\Filament\Resources\AuthorResource\Pages\ListAuthors;
use MbeGroup\Author\Filament\Resources\AuthorResource\Pages\CreateAuthor;
use MbeGroup\Author\Filament\Resources\AuthorResource\Pages\EditAuthor;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

class AuthorResourcePagesTest extends TestCase
{
    /**
     * Test sprawdzający czy ListAuthors można zinstancjonować
     */
    public function test_can_instantiate_list_authors(): void
    {
        $reflection = new ReflectionClass(ListAuthors::class);

        $this->assertEquals('ListAuthors', $reflection->getShortName());
        $this->assertEquals('MbeGroup\Author\Filament\Resources\AuthorResource\Pages', $reflection->getNamespaceName());
    }

    /**
     * Test sprawdzający czy CreateAuthor można zinstancjonować
     */
    public function test_can_instantiate_create_author(): void
    {
        $reflection = new ReflectionClass(CreateAuthor::class);

        $this->assertEquals('CreateAuthor', $reflection->getShortName());
        $this->assertEquals('MbeGroup\Author\Filament\Resources\AuthorResource\Pages', $reflection->getNamespaceName());
    }

    /**
     * Test sprawdzający czy EditAuthor można zinstancjonować
     */
    public function test_can_instantiate_edit_author(): void
    {
        $reflection = new ReflectionClass(EditAuthor::class);

        $this->assertEquals('EditAuthor', $reflection->getShortName());
        $this->assertEquals('MbeGroup\Author\Filament\Resources\AuthorResource\Pages', $reflection->getNamespaceName());
    }

    /**
     * Test sprawdzający czy ListAuthors dziedziczy po odpowiedniej klasie
     */
    public function test_list_authors_extends_correct_class(): void
    {
        $reflection = new ReflectionClass(ListAuthors::class);

        // Sprawdzenie czy dziedziczy po ListRecords
        $this->assertTrue($reflection->isSubclassOf(\Filament\Resources\Pages\ListRecords::class));
    }

    /**
     * Test sprawdzający czy CreateAuthor dziedziczy po odpowiedniej klasie
     */
    public function test_create_author_extends_correct_class(): void
    {
        $reflection = new ReflectionClass(CreateAuthor::class);

        // Sprawdzenie czy dziedziczy po CreateRecord
        $this->assertTrue($reflection->isSubclassOf(\Filament\Resources\Pages\CreateRecord::class));
    }

    /**
     * Test sprawdzający czy EditAuthor dziedziczy po odpowiedniej klasie
     */
    public function test_edit_author_extends_correct_class(): void
    {
        $reflection = new ReflectionClass(EditAuthor::class);

        // Sprawdzenie czy dziedziczy po EditRecord
        $this->assertTrue($reflection->isSubclassOf(\Filament\Resources\Pages\EditRecord::class));
    }

    /**
     * Test sprawdzający czy wszystkie strony mają właściwość $resource
     */
    public function test_all_pages_have_resource_property(): void
    {
        $pages = [
            ListAuthors::class,
            CreateAuthor::class,
            EditAuthor::class,
        ];

        foreach ($pages as $pageClass) {
            $reflection = new ReflectionClass($pageClass);

            $this->assertTrue(
                $reflection->hasProperty('resource'),
                "Strona {$pageClass} powinna mieć właściwość \$resource"
            );
        }
    }

    /**
     * Test sprawdzający czy wszystkie klasy stron są publiczne
     */
    public function test_all_page_classes_are_public(): void
    {
        $pages = [
            ListAuthors::class,
            CreateAuthor::class,
            EditAuthor::class,
        ];

        foreach ($pages as $pageClass) {
            $reflection = new ReflectionClass($pageClass);

            // Klasy PHP domyślnie są publiczne, sprawdzamy czy nie są abstract
            $this->assertFalse(
                $reflection->isAbstract(),
                "Klasa {$pageClass} nie powinna być abstrakcyjna"
            );
        }
    }

    /**
     * Test sprawdzający czy strony nie są abstrakcyjne
     */
    public function test_page_classes_are_not_abstract(): void
    {
        $pages = [
            ListAuthors::class,
            CreateAuthor::class,
            EditAuthor::class,
        ];

        foreach ($pages as $pageClass) {
            $reflection = new ReflectionClass($pageClass);

            $this->assertFalse(
                $reflection->isAbstract(),
                "Klasa {$pageClass} nie powinna być abstrakcyjna"
            );
        }
    }

    /**
     * Test sprawdzający czy strony nie są finalne
     */
    public function test_page_classes_are_not_final(): void
    {
        $pages = [
            ListAuthors::class,
            CreateAuthor::class,
            EditAuthor::class,
        ];

        foreach ($pages as $pageClass) {
            $reflection = new ReflectionClass($pageClass);

            $this->assertFalse(
                $reflection->isFinal(),
                "Klasa {$pageClass} nie powinna być finalna"
            );
        }
    }

    /**
     * Test sprawdzający czy klasy mają konstruktory lub mogą być instancjonowane
     */
    public function test_page_classes_can_be_instantiated_or_have_constructors(): void
    {
        $pages = [
            ListAuthors::class,
            CreateAuthor::class,
            EditAuthor::class,
        ];

        foreach ($pages as $pageClass) {
            $reflection = new ReflectionClass($pageClass);

            $this->assertTrue(
                $reflection->isInstantiable() || $reflection->hasMethod('__construct'),
                "Klasa {$pageClass} powinna być instancjonowalna lub mieć konstruktor"
            );
        }
    }

    /**
     * Test sprawdzający czy wszystkie strony należą do tego samego namespace
     */
    public function test_all_pages_share_same_namespace(): void
    {
        $expectedNamespace = 'MbeGroup\Author\Filament\Resources\AuthorResource\Pages';

        $pages = [
            ListAuthors::class,
            CreateAuthor::class,
            EditAuthor::class,
        ];

        foreach ($pages as $pageClass) {
            $reflection = new ReflectionClass($pageClass);

            $this->assertEquals(
                $expectedNamespace,
                $reflection->getNamespaceName(),
                "Klasa {$pageClass} powinna być w namespace {$expectedNamespace}"
            );
        }
    }

    /**
     * Test sprawdzający nazwy klas
     */
    public function test_page_class_names(): void
    {
        $expectedNames = [
            ListAuthors::class => 'ListAuthors',
            CreateAuthor::class => 'CreateAuthor',
            EditAuthor::class => 'EditAuthor',
        ];

        foreach ($expectedNames as $class => $expectedName) {
            $reflection = new ReflectionClass($class);

            $this->assertEquals(
                $expectedName,
                $reflection->getShortName(),
                "Klasa {$class} powinna mieć nazwę {$expectedName}"
            );
        }
    }
}
