<?php
namespace MbeGroup\Author\Models;

use MongoDB\Laravel\Eloquent\Model;
use App\ValueObject\Status;
use App\Models\Dictionary;
use Illuminate\Support\Facades\Cache;

class Author extends Model
{
    protected $connection = 'mongodb';
    protected $collection = 'authors';
    protected $fillable   = [
        'name',
        'slug',
        'position',
        'company_name',
        'type_id',
        'email',
        'image',
        'biography',
        'status',
    ];

    protected $casts = [
        'status'    => Status::class,
    ];

    public function type()
    {
        return $this->belongsTo(Dictionary::class, 'type_id');
    }

    /**
     * Pobiera cache'owane opcje selecta
     */
    public static function getSelectOptions(?int $typeId = null): array
    {
        $cacheKey = $typeId ? "author_select_options_type_{$typeId}" : 'author_select_options';

        return Cache::remember($cacheKey, 3600, function () use ($typeId) {
            $query = self::where('status', Status::ACTIVE);

            if ($typeId) {
                $query->where('type_id', $typeId);
            }

            return $query->orderBy('name')
                ->get()
                ->pluck('name', '_id')
                ->toArray();
        });
    }

    /**
     * Wyszukuje autorów do selecta
     */
    public static function searchForSelect(string $search, ?int $typeId = null): array
    {
        $cacheKey = 'author_search_select_' . md5($search . ($typeId ? "_type_{$typeId}" : ''));

        return Cache::remember($cacheKey, 3600, function () use ($search, $typeId) {
            $query = self::where('name', 'like', '%' . $search . '%')
                ->where('status', Status::ACTIVE);

            if ($typeId) {
                $query->where('type_id', $typeId);
            }

            return $query->orderBy('name')
                ->get()
                ->pluck('name', '_id')
                ->toArray();
        });
    }

    /**
     * Czyści cache opcji selectów
     */
    public static function clearSelectCache(): void
    {
        Cache::forget('author_select_options');

        $types = Dictionary::where('type', 'author_type')->pluck('_id');
        foreach ($types as $typeId) {
            Cache::forget("author_select_options_type_{$typeId}");
        }
    }

    public function getImageUrlAttribute(): ?string
    {
        return $this->image ? url('storage/' . ltrim($this->image, '/')) : null;
    }
}
