<?php

namespace Tests\Unit;

use MbeGroup\Author\Providers\AuthorServiceProvider;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

class AuthorServiceProviderTest extends TestCase
{
    /**
     * Test sprawdzający czy AuthorServiceProvider można zinstancjonować
     */
    public function test_can_instantiate_service_provider(): void
    {
        $reflection = new ReflectionClass(AuthorServiceProvider::class);

        $this->assertEquals('AuthorServiceProvider', $reflection->getShortName());
        $this->assertEquals('MbeGroup\Author\Providers', $reflection->getNamespaceName());
    }

    /**
     * Test sprawdzający czy klasa dziedziczy po ServiceProvider
     */
    public function test_extends_service_provider(): void
    {
        $reflection = new ReflectionClass(AuthorServiceProvider::class);

        // Sprawdzenie czy dziedziczy po Illuminate ServiceProvider
        $this->assertTrue($reflection->isSubclassOf(\Illuminate\Support\ServiceProvider::class));
    }

    /**
     * Test sprawdzający czy klasa jest finalna
     */
    public function test_class_is_final(): void
    {
        $reflection = new ReflectionClass(AuthorServiceProvider::class);

        $this->assertTrue($reflection->isFinal(), 'AuthorServiceProvider powinien być klasą finalną');
    }

    /**
     * Test sprawdzający czy ma metodę boot
     */
    public function test_has_boot_method(): void
    {
        $reflection = new ReflectionClass(AuthorServiceProvider::class);

        $this->assertTrue($reflection->hasMethod('boot'), 'ServiceProvider powinien mieć metodę boot');

        $bootMethod = $reflection->getMethod('boot');
        $this->assertTrue($bootMethod->isPublic(), 'Metoda boot powinna być publiczna');
    }

    /**
     * Test sprawdzający parametry metody boot
     */
    public function test_boot_method_parameters(): void
    {
        $reflection = new ReflectionClass(AuthorServiceProvider::class);
        $bootMethod = $reflection->getMethod('boot');

        $parameters = $bootMethod->getParameters();
        $this->assertCount(0, $parameters, 'Metoda boot nie powinna mieć parametrów');
    }

    /**
     * Test sprawdzający return type metody boot
     */
    public function test_boot_method_return_type(): void
    {
        $reflection = new ReflectionClass(AuthorServiceProvider::class);
        $bootMethod = $reflection->getMethod('boot');

        $returnType = $bootMethod->getReturnType();
        $this->assertNotNull($returnType, 'Metoda boot powinna mieć zdefiniowany return type');
        $this->assertEquals('void', $returnType->getName(), 'Metoda boot powinna zwracać void');
    }

    /**
     * Test sprawdzający czy klasa nie ma niepotrzebnych publicznych właściwości
     */
    public function test_no_unexpected_public_properties(): void
    {
        $reflection = new ReflectionClass(AuthorServiceProvider::class);
        $publicProperties = [];

        foreach ($reflection->getProperties(\ReflectionProperty::IS_PUBLIC) as $property) {
            if ($property->class === AuthorServiceProvider::class) {
                $publicProperties[] = $property->name;
            }
        }

        // ServiceProvider nie powinien mieć żadnych dodatkowych publicznych właściwości
        $this->assertEmpty($publicProperties, 'ServiceProvider nie powinien mieć dodatkowych publicznych właściwości');
    }

    /**
     * Test sprawdzający czy klasa nie ma konstruktora lub ma domyślny konstruktor
     */
    public function test_constructor_configuration(): void
    {
        $reflection = new ReflectionClass(AuthorServiceProvider::class);

        if ($reflection->hasMethod('__construct')) {
            $constructor = $reflection->getMethod('__construct');
            $this->assertTrue($constructor->isPublic(), 'Konstruktor powinien być publiczny');
        } else {
            // Jeśli nie ma konstruktora, to używa domyślnego z klasy nadrzędnej
            $this->assertTrue(true, 'Klasa używa domyślnego konstruktora');
        }
    }

    /**
     * Test sprawdzający czy nie ma metody register lub ma odpowiednią implementację
     */
    public function test_register_method_configuration(): void
    {
        $reflection = new ReflectionClass(AuthorServiceProvider::class);

        if ($reflection->hasMethod('register')) {
            $registerMethod = $reflection->getMethod('register');
            $this->assertTrue($registerMethod->isPublic(), 'Metoda register powinna być publiczna');

            $returnType = $registerMethod->getReturnType();
            if ($returnType !== null) {
                $this->assertEquals('void', $returnType->getName(), 'Metoda register powinna zwracać void');
            }
        } else {
            // Jeśli nie ma metody register, to używa domyślnej z klasy nadrzędnej
            $this->assertTrue(true, 'Klasa używa domyślnej implementacji metody register');
        }
    }

    /**
     * Test sprawdzający podstawowe właściwości klasy
     */
    public function test_basic_class_properties(): void
    {
        $reflection = new ReflectionClass(AuthorServiceProvider::class);

        // Klasy PHP domyślnie są publiczne, sprawdzamy główne właściwości
        $this->assertFalse($reflection->isAbstract(), 'Klasa nie powinna być abstrakcyjna');
        $this->assertTrue($reflection->isInstantiable(), 'Klasa powinna być instancjonowalna');
        $this->assertTrue($reflection->isFinal(), 'Klasa powinna być finalna');
    }

    /**
     * Test sprawdzający czy nie ma dodatkowych niezdefiniowanych metod
     */
    public function test_no_unexpected_methods(): void
    {
        $reflection = new ReflectionClass(AuthorServiceProvider::class);
        $ownMethods = [];

        foreach ($reflection->getMethods() as $method) {
            if ($method->class === AuthorServiceProvider::class) {
                $ownMethods[] = $method->name;
            }
        }

        // Sprawdzenie czy są tylko oczekiwane metody
        $allowedMethods = ['boot', 'register', '__construct'];
        foreach ($ownMethods as $method) {
            $this->assertContains($method, $allowedMethods,
                "Metoda {$method} nie jest oczekiwaną metodą w ServiceProvider");
        }
    }

    /**
     * Test sprawdzający namespace consistency
     */
    public function test_namespace_consistency(): void
    {
        $reflection = new ReflectionClass(AuthorServiceProvider::class);

        $this->assertStringStartsWith('MbeGroup\Author', $reflection->getNamespaceName(),
            'ServiceProvider powinien być w przestrzeni nazw MbeGroup\Author');
    }
}
