<?php

namespace MbeGroup\Content\Tests\Unit\Models;

use MbeGroup\Content\Models\Content;
use PHPUnit\Framework\TestCase;
use ReflectionClass;
use App\ValueObject\Status;

class ContentCastingTest extends TestCase
{
    /**
     * Test sprawdzający konfigurację castów
     */
    public function test_casts_configuration(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $castsProperty = $reflection->getProperty('casts');
        $castsProperty->setAccessible(true);
        $casts = $castsProperty->getValue($content);

        $expectedCasts = [
            'status'             => Status::class,
            'industry_ids'       => 'array',
            'specialization_ids' => 'array',
            'tags'               => 'array',
            'location'           => 'array',
            'expert_ids'         => 'array',
            'author_ids'         => 'array',
        ];

        $this->assertEquals($expectedCasts, $casts);
    }

    /**
     * Test sprawdzający poprawność castowania statusu
     */
    public function test_status_cast_to_value_object(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $castsProperty = $reflection->getProperty('casts');
        $castsProperty->setAccessible(true);
        $casts = $castsProperty->getValue($content);

        $this->assertEquals(Status::class, $casts['status']);
    }

    /**
     * Test sprawdzający poprawność castowania tablic
     */
    public function test_array_casts(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $castsProperty = $reflection->getProperty('casts');
        $castsProperty->setAccessible(true);
        $casts = $castsProperty->getValue($content);

        $expectedArrayCasts = ['industry_ids', 'specialization_ids', 'tags', 'location', 'expert_ids', 'author_ids'];

        foreach ($expectedArrayCasts as $arrayField) {
            $this->assertEquals('array', $casts[$arrayField], "Pole '{$arrayField}' powinno być castowane na array");
        }
    }

    /**
     * Test sprawdzający czy wszystkie pola w casts są również w fillable
     */
    public function test_all_cast_fields_are_fillable(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $castsProperty = $reflection->getProperty('casts');
        $castsProperty->setAccessible(true);
        $casts = $castsProperty->getValue($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        foreach (array_keys($casts) as $castField) {
            $this->assertContains(
                $castField,
                $fillable,
                "Pole '{$castField}' z casts powinno być również w fillable"
            );
        }
    }

    /**
     * Test sprawdzający integralność danych - czy wszystkie ID fields są w array casts
     */
    public function test_id_arrays_are_properly_cast(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $castsProperty = $reflection->getProperty('casts');
        $castsProperty->setAccessible(true);
        $casts = $castsProperty->getValue($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        $idFields = array_filter($fillable, function($field) {
            return str_ends_with($field, '_ids');
        });

        foreach ($idFields as $idField) {
            $this->assertArrayHasKey(
                $idField,
                $casts,
                "Pole ID array '{$idField}' powinno być w casts"
            );
            $this->assertEquals(
                'array',
                $casts[$idField],
                "Pole ID array '{$idField}' powinno być castowane na array"
            );
        }
    }
}
