<?php

namespace MbeGroup\Content\Tests\Unit\Models;

use MbeGroup\Content\Models\Content;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

class ContentModelTest extends TestCase
{
    /**
     * Test tworzenia nowej instancji Content
     */
    public function test_can_create_content_instance(): void
    {
        $content = new Content();

        $this->assertInstanceOf(Content::class, $content);
    }

    /**
     * Test sprawdzający czy klasa dziedziczy po MongoDB Model
     */
    public function test_extends_mongodb_model(): void
    {
        $content = new Content();

        $this->assertInstanceOf(\MongoDB\Laravel\Eloquent\Model::class, $content);
    }

    /**
     * Test sprawdzający czy klasa używa HasSeoFields trait
     */
    public function test_uses_has_seo_fields_trait(): void
    {
        $reflection = new ReflectionClass(Content::class);
        $traits = $reflection->getTraitNames();

        $this->assertContains('App\Traits\HasSeoFields', $traits);
    }

    /**
     * Test sprawdzający konfigurację połączenia MongoDB
     */
    public function test_mongodb_connection_configuration(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $connectionProperty = $reflection->getProperty('connection');
        $connectionProperty->setAccessible(true);
        $connection = $connectionProperty->getValue($content);

        $this->assertEquals('mongodb', $connection);
    }

    /**
     * Test sprawdzający nazwę kolekcji MongoDB
     */
    public function test_mongodb_collection_name(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $collectionProperty = $reflection->getProperty('collection');
        $collectionProperty->setAccessible(true);
        $collection = $collectionProperty->getValue($content);

        $this->assertEquals('contents', $collection);
    }

    /**
     * Test sprawdzający wypełnialne pola
     */
    public function test_fillable_attributes(): void
    {
        $content = new Content();
        $fillable = $content->getFillable();

        $expectedFillable = [
            'title',
            'subtitle',
            'slug',
            'short_title',
            'content_category_id',
            'industry_ids',
            'specialization_ids',
            'employer_id',
            'location',
            'is_sponsored',
            'is_sticky',
            'main_image',
            'main_image_caption',
            'status',
            'type',
            'lead',
            'content',
            'tags',
            'publication_date',
            'author_ids',
            'expert_ids',
            'employer_image',
            'project_type',
            'project_scope',
            'project_duration',
            'people_involved_count',
            'key_numbers',
            'quotes',
            'gallery',
            'team_description',
            'involved_people',
            'team_composition',
            'software',
            'hardware',
            'methodologies',
            'seo_title',
            'seo_description',
            'seo_keywords',
            'seo_noindex',
        ];

        $this->assertEquals($expectedFillable, $fillable);
    }

    /**
     * Test konstruktora z danymi
     */
    public function test_constructor_with_attributes(): void
    {
        $attributes = [
            'title' => 'Test Article',
            'slug' => 'test-article',
            'short_title' => 'Test',
            'type' => 'article',
            'status' => 'published',
            'is_sponsored' => false,
            'is_sticky' => false,
        ];

        $content = new Content($attributes);
        $this->assertInstanceOf(Content::class, $content);
    }

    /**
     * Test sprawdzający czy konstruktor nie rzuca wyjątków z pustymi danymi
     */
    public function test_constructor_with_empty_attributes(): void
    {
        $content = new Content([]);
        $this->assertInstanceOf(Content::class, $content);
    }

    /**
     * Test sprawdzający czy konstruktor działa z domyślnymi parametrami
     */
    public function test_constructor_with_default_parameters(): void
    {
        $content = new Content();
        $this->assertInstanceOf(Content::class, $content);
    }

    /**
     * Test sprawdzający czy wszystkie wymagane pola biznesowe są wypełnialne
     */
    public function test_required_business_fields_are_fillable(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        $requiredBusinessFields = ['title', 'slug', 'type', 'status'];

        foreach ($requiredBusinessFields as $field) {
            $this->assertContains(
                $field,
                $fillable,
                "Wymagane pole biznesowe '{$field}' nie jest w fillable"
            );
        }
    }

    /**
     * Test sprawdzający czy nie ma zduplikowanych pól w fillable
     */
    public function test_no_duplicate_fillable_fields(): void
    {
        $content = new Content();
        $fillable = $content->getFillable();

        $this->assertEquals(
            count($fillable),
            count(array_unique($fillable)),
            'Nie powinno być zduplikowanych pól w fillable'
        );
    }

    /**
     * Test sprawdzający czy wszystkie pola fillable są stringami
     */
    public function test_fillable_field_types(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        foreach ($fillable as $field) {
            $this->assertIsString($field, "Pole fillable '{$field}' powinno być stringiem");
        }
    }

    /**
     * Test sprawdzający namespace klasy
     */
    public function test_class_namespace(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $this->assertEquals('MbeGroup\Content\Models', $reflection->getNamespaceName());
        $this->assertEquals('Content', $reflection->getShortName());
    }

    /**
     * Test sprawdzający scope ofType
     */
    public function test_scope_of_type(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $this->assertTrue($reflection->hasMethod('scopeOfType'));

        $method = $reflection->getMethod('scopeOfType');
        $this->assertTrue($method->isPublic());
        $this->assertCount(2, $method->getParameters());

        $parameters = $method->getParameters();
        $this->assertEquals('query', $parameters[0]->getName());
        $this->assertEquals('type', $parameters[1]->getName());
    }

    /**
     * Test sprawdzający czy scope ofType zwraca Builder
     */
    public function test_scope_of_type_returns_builder(): void
    {
        $reflection = new ReflectionClass(Content::class);
        $method = $reflection->getMethod('scopeOfType');

        $this->assertNotNull($method->getReturnType());
        $this->assertEquals('Illuminate\Database\Eloquent\Builder', $method->getReturnType()->getName());
    }
}
