<?php

namespace MbeGroup\Content\Tests\Unit\Models;

use MbeGroup\Content\Models\Content;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

class ContentValidationTest extends TestCase
{
    /**
     * Test sprawdzający walidację wymaganych pól dla artykułu
     */
    public function test_article_required_fields(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        $articleRequiredFields = ['title', 'slug', 'type', 'status', 'content'];

        foreach ($articleRequiredFields as $field) {
            $this->assertContains(
                $field,
                $fillable,
                "Wymagane pole dla artykułu '{$field}' nie jest w fillable"
            );
        }
    }

    /**
     * Test sprawdzający czy pola związane z publikacją są fillable
     */
    public function test_publication_fields_are_fillable(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        $publicationFields = ['publication_date', 'is_sponsored', 'is_sticky'];

        foreach ($publicationFields as $field) {
            $this->assertContains(
                $field,
                $fillable,
                "Pole publikacji '{$field}' powinno być w fillable"
            );
        }
    }

    /**
     * Test sprawdzający czy pola związane z obrazami są fillable
     */
    public function test_image_fields_are_fillable(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        $imageFields = ['main_image', 'main_image_caption'];

        foreach ($imageFields as $field) {
            $this->assertContains(
                $field,
                $fillable,
                "Pole obrazu '{$field}' powinno być w fillable"
            );
        }
    }

    /**
     * Test sprawdzający czy pola SEO są dostępne przez trait
     */
    public function test_seo_fields_from_trait(): void
    {
        $content = new Content();

        // Sprawdzenie czy trait HasSeoFields dodaje wymagane metody
        $reflection = new ReflectionClass($content);
        $traits = $reflection->getTraitNames();

        $this->assertContains('App\Traits\HasSeoFields', $traits);
    }

    /**
     * Test sprawdzający czy pola związane z kategoryzacją są fillable
     */
    public function test_categorization_fields_are_fillable(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        $categorizationFields = ['content_category_id', 'industry_ids', 'specialization_ids', 'tags'];

        foreach ($categorizationFields as $field) {
            $this->assertContains(
                $field,
                $fillable,
                "Pole kategoryzacji '{$field}' powinno być w fillable"
            );
        }
    }

    /**
     * Test sprawdzający czy pola związane z autorami są fillable
     */
    public function test_author_fields_are_fillable(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        $authorFields = ['author_ids', 'expert_ids'];

        foreach ($authorFields as $field) {
            $this->assertContains(
                $field,
                $fillable,
                "Pole autorów '{$field}' powinno być w fillable"
            );
        }
    }

    /**
     * Test sprawdzający czy pola lokalizacji są fillable
     */
    public function test_location_fields_are_fillable(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        $this->assertContains('location', $fillable, "Pole 'location' powinno być w fillable");
    }

    /**
     * Test sprawdzający czy pole employer_id jest fillable
     */
    public function test_employer_field_is_fillable(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        $this->assertContains('employer_id', $fillable, "Pole 'employer_id' powinno być w fillable");
    }

    /**
     * Test sprawdzający czy pola SEO są fillable
     */
    public function test_seo_fields_are_fillable(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        $seoFields = ['seo_title', 'seo_description', 'seo_keywords', 'seo_noindex'];

        foreach ($seoFields as $field) {
            $this->assertContains(
                $field,
                $fillable,
                "Pole SEO '{$field}' powinno być w fillable"
            );
        }
    }
}
