<?php

namespace MbeGroup\Content\Tests\Unit;

use MbeGroup\Content\Models\Content;
use PHPUnit\Framework\TestCase;
use ReflectionClass;
use App\ValueObject\Status;
use App\Models\Dictionary;
use MbeGroup\Employer\Models\Employer;
use MbeGroup\Author\Models\Author;

class ContentTest extends TestCase
{
    /**
     * Test tworzenia nowej instancji Content
     */
    public function test_can_create_content_instance(): void
    {
        $content = new Content();

        $this->assertInstanceOf(Content::class, $content);
    }

    /**
     * Test sprawdzający czy klasa dziedziczy po MongoDB Model
     */
    public function test_extends_mongodb_model(): void
    {
        $content = new Content();

        $this->assertInstanceOf(\MongoDB\Laravel\Eloquent\Model::class, $content);
    }

    /**
     * Test sprawdzający czy klasa używa HasSeoFields trait
     */
    public function test_uses_has_seo_fields_trait(): void
    {
        $reflection = new ReflectionClass(Content::class);
        $traits = $reflection->getTraitNames();

        $this->assertContains('App\Traits\HasSeoFields', $traits);
    }

    /**
     * Test sprawdzający konfigurację połączenia MongoDB
     */
    public function test_mongodb_connection_configuration(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $connectionProperty = $reflection->getProperty('connection');
        $connectionProperty->setAccessible(true);
        $connection = $connectionProperty->getValue($content);

        $this->assertEquals('mongodb', $connection);
    }

    /**
     * Test sprawdzający nazwę kolekcji MongoDB
     */
    public function test_mongodb_collection_name(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $collectionProperty = $reflection->getProperty('collection');
        $collectionProperty->setAccessible(true);
        $collection = $collectionProperty->getValue($content);

        $this->assertEquals('contents', $collection);
    }

    /**
     * Test sprawdzający wypełnialne pola
     */
    public function test_fillable_attributes(): void
    {
        $content = new Content();
        $fillable = $content->getFillable();

        $expectedFillable = [
            'title',
            'subtitle',
            'slug',
            'short_title',
            'content_category_id',
            'industry_ids',
            'specialization_ids',
            'employer_id',
            'location',
            'is_sponsored',
            'is_sticky',
            'main_image',
            'main_image_caption',
            'status',
            'type',
            'lead',
            'content',
            'tags',
            'publication_date',
            'author_ids',
            'expert_ids',
            'employer_image',
            'project_type',
            'project_scope',
            'project_duration',
            'people_involved_count',
            'key_numbers',
            'gallery',
            'team_description',
            'involved_people',
            'team_composition',
            'software',
            'hardware',
            'methodologies',
            'seo_title',
            'seo_description',
            'seo_keywords',
            'seo_noindex',
            'seo_opengraph_image',
        ];

        $this->assertEquals($expectedFillable, $fillable);
    }

    /**
     * Test sprawdzający konfigurację castów
     */
    public function test_casts_configuration(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $castsProperty = $reflection->getProperty('casts');
        $castsProperty->setAccessible(true);
        $casts = $castsProperty->getValue($content);

        $expectedCasts = [
            'status'           => Status::class,
            'gallery'          => 'array',
            'publication_date' => 'datetime',
        ];

        $this->assertEquals($expectedCasts, $casts);
    }

    /**
     * Test sprawdzający relację z kategorią treści
     */
    public function test_content_category_relationship(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $this->assertTrue($reflection->hasMethod('contentCategory'));

        $method = $reflection->getMethod('contentCategory');
        $this->assertTrue($method->isPublic());
        $this->assertCount(0, $method->getParameters());
    }

    /**
     * Test sprawdzający relację z pracodawcą
     */
    public function test_employer_relationship(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $this->assertTrue($reflection->hasMethod('employer'));

        $method = $reflection->getMethod('employer');
        $this->assertTrue($method->isPublic());
        $this->assertCount(0, $method->getParameters());
    }

    /**
     * Test sprawdzający relację z branżami
     */
    public function test_industries_relationship(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

    $this->assertTrue($reflection->hasMethod('getIndustries'));
    $method = $reflection->getMethod('getIndustries');
    $this->assertTrue($method->isPublic());
    }

    /**
     * Test sprawdzający relację ze specjalizacjami
     */
    public function test_specializations_relationship(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

    $this->assertTrue($reflection->hasMethod('getSpecializations'));
    $method = $reflection->getMethod('getSpecializations');
    $this->assertTrue($method->isPublic());
    }

    /**
     * Test sprawdzający relację z ekspertami
     */
    public function test_experts_relationship(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $this->assertTrue($reflection->hasMethod('experts'));

        $method = $reflection->getMethod('experts');
        $this->assertTrue($method->isPublic());
        $this->assertCount(0, $method->getParameters());
    }

    /**
     * Test sprawdzający relację z autorami
     */
    public function test_authors_relationship(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

    $this->assertTrue($reflection->hasMethod('getAuthorsListAttribute'));
    $method = $reflection->getMethod('getAuthorsListAttribute');
    $this->assertTrue($method->isPublic());
    }

    /**
     * Test sprawdzający scope ofType
     */
    public function test_scope_of_type(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $this->assertTrue($reflection->hasMethod('scopeOfType'));

        $method = $reflection->getMethod('scopeOfType');
        $this->assertTrue($method->isPublic());
        $this->assertCount(2, $method->getParameters());

        $parameters = $method->getParameters();
        $this->assertEquals('query', $parameters[0]->getName());
        $this->assertEquals('type', $parameters[1]->getName());
    }

    /**
     * Test konstruktora z danymi
     */
    public function test_constructor_with_attributes(): void
    {
        $expectedFillable = [
            'title',
            'subtitle',
            'slug',
            'short_title',
            'content_category_id',
            'industry_ids',
            'specialization_ids',
            'employer_id',
            'location',
            'is_sponsored',
            'is_sticky',
            'main_image',
            'main_image_caption',
            'status',
            'type',
            'lead',
            'content',
            'tags',
            'publication_date',
            'author_ids',
            'expert_ids',
            'employer_image',
            'project_type',
            'project_scope',
            'project_duration',
            'people_involved_count',
            'key_numbers',
            'gallery',
            'team_description',
            'involved_people',
            'team_composition',
            'software',
            'hardware',
            'methodologies',
        ];
        $content = new Content();
        $fillable = $content->getFillable();

        $this->assertEquals(
            count($fillable),
            count(array_unique($fillable)),
            'Nie powinno być zduplikowanych pól w fillable'
        );
    }

    /**
     * Test sprawdzający czy wszystkie pola fillable są stringami
     */
    public function test_fillable_field_types(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        foreach ($fillable as $field) {
            $this->assertIsString($field, "Pole fillable '{$field}' powinno być stringiem");
        }
    }

    /**
     * Test sprawdzający namespace klasy
     */
    public function test_class_namespace(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $this->assertEquals('MbeGroup\Content\Models', $reflection->getNamespaceName());
        $this->assertEquals('Content', $reflection->getShortName());
    }

    /**
     * Test sprawdzający poprawność castowania statusu
     */
    public function test_status_cast_to_value_object(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $castsProperty = $reflection->getProperty('casts');
        $castsProperty->setAccessible(true);
        $casts = $castsProperty->getValue($content);

        $this->assertEquals(Status::class, $casts['status']);
    }

    /**
     * Test sprawdzający poprawność castowania tablic
     */
    public function test_array_casts(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $castsProperty = $reflection->getProperty('casts');
        $castsProperty->setAccessible(true);
        $casts = $castsProperty->getValue($content);

        $expectedArrayCasts = ['gallery'];

        foreach ($expectedArrayCasts as $arrayField) {
            $this->assertEquals('array', $casts[$arrayField], "Pole '{$arrayField}' powinno być castowane na array");
        }
    }

    /**
     * Test sprawdzający czy wszystkie pola w casts są również w fillable
     */
    public function test_all_cast_fields_are_fillable(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $castsProperty = $reflection->getProperty('casts');
        $castsProperty->setAccessible(true);
        $casts = $castsProperty->getValue($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        foreach (array_keys($casts) as $castField) {
            $this->assertContains(
                $castField,
                $fillable,
                "Pole '{$castField}' z casts powinno być również w fillable"
            );
        }
    }

    /**
     * Test sprawdzający czy scope ofType zwraca Builder
     */
    public function test_scope_of_type_returns_builder(): void
    {
        $reflection = new ReflectionClass(Content::class);
        $method = $reflection->getMethod('scopeOfType');

        $this->assertNotNull($method->getReturnType());
        $this->assertEquals('Illuminate\Database\Eloquent\Builder', $method->getReturnType()->getName());
    }

    /**
     * Test sprawdzający czy wszystkie metody relacyjne istnieją
     */
    public function test_all_relationship_methods_exist(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $expectedRelationships = [
            'contentCategory',
            'employer',
            'getIndustries',
            'getSpecializations',
            'experts',
            'getAuthorsListAttribute'
        ];

        foreach ($expectedRelationships as $relationship) {
            $this->assertTrue(
                $reflection->hasMethod($relationship),
                "Metoda relacji '{$relationship}' nie została znaleziona"
            );

            $method = $reflection->getMethod($relationship);
            $this->assertTrue($method->isPublic(), "Metoda '{$relationship}' powinna być publiczna");
        }
    }

    /**
     * Test sprawdzający czy pola SEO są dostępne przez trait
     */
    public function test_seo_fields_from_trait(): void
    {
        $content = new Content();

        // Sprawdzenie czy trait HasSeoFields dodaje wymagane metody
        $reflection = new ReflectionClass($content);
        $traits = $reflection->getTraitNames();

        $this->assertContains('App\Traits\HasSeoFields', $traits);
    }

    /**
     * Test sprawdzający czy konstruktor nie rzuca wyjątków z pustymi danymi
     */
    public function test_constructor_with_empty_attributes(): void
    {
        $content = new Content([]);
        $this->assertInstanceOf(Content::class, $content);
    }

    /**
     * Test sprawdzający czy konstruktor działa z domyślnymi parametrami
     */
    public function test_constructor_with_default_parameters(): void
    {
        $content = new Content();
        $this->assertInstanceOf(Content::class, $content);
    }

    /**
     * Test sprawdzający walidację wymaganych pól dla artykułu
     */
    public function test_article_required_fields(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        $articleRequiredFields = ['title', 'slug', 'type', 'status', 'content'];

        foreach ($articleRequiredFields as $field) {
            $this->assertContains(
                $field,
                $fillable,
                "Wymagane pole dla artykułu '{$field}' nie jest w fillable"
            );
        }
    }

    /**
     * Test sprawdzający czy pola związane z publikacją są fillable
     */
    public function test_publication_fields_are_fillable(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        $publicationFields = ['publication_date', 'is_sponsored', 'is_sticky'];

        foreach ($publicationFields as $field) {
            $this->assertContains(
                $field,
                $fillable,
                "Pole publikacji '{$field}' powinno być w fillable"
            );
        }
    }

    /**
     * Test sprawdzający czy pola związane z obrazami są fillable
     */
    public function test_image_fields_are_fillable(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($content);

        $imageFields = ['main_image', 'main_image_caption'];

        foreach ($imageFields as $field) {
            $this->assertContains(
                $field,
                $fillable,
                "Pole obrazu '{$field}' powinno być w fillable"
            );
        }
    }

    /**
     * Test sprawdzający czy metody relacji belongsTo mają odpowiednie parametry
     */
    public function test_belongs_to_relationship_parameters(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $belongsToMethods = ['contentCategory', 'employer'];

        foreach ($belongsToMethods as $method) {
            $relationshipMethod = $reflection->getMethod($method);
            $this->assertCount(
                0,
                $relationshipMethod->getParameters(),
                "Metoda belongsTo '{$method}' nie powinna mieć parametrów"
            );
        }
    }

    /**
     * Test sprawdzający czy metody relacji belongsToMany mają odpowiednie parametry
     */
    public function test_belongs_to_many_relationship_parameters(): void
    {
        $content = new Content();
        $reflection = new ReflectionClass($content);

        $belongsToManyMethods = ['experts'];
        foreach ($belongsToManyMethods as $method) {
            $relationshipMethod = $reflection->getMethod($method);
            $this->assertCount(
                0,
                $relationshipMethod->getParameters(),
                "Metoda belongsToMany '{$method}' nie powinna mieć parametrów"
            );
        }
    }
}
