<?php

namespace MbeGroup\Content\Services;

use App\Helpers\DictionaryHelper;
use MbeGroup\Content\Repository\ContentRepository;
use MbeGroup\Content\Models\Content;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use MbeGroup\Content\Contracts\ContentServiceInterface;
use MbeGroup\Content\Filament\Resources\ArticleResource;
use MbeGroup\Content\Filament\Clusters\KnowledgeBase\Resources\QualificationResource;
use MbeGroup\Content\Filament\Clusters\KnowledgeBase\Resources\JobPositionDescriptionResource;
use MbeGroup\Content\Filament\Clusters\KnowledgeBase\Resources\TechnologyToolResource;
use MbeGroup\Content\Filament\Clusters\KnowledgeBase\Resources\DictionaryEntryResource;
use MbeGroup\Content\Filament\Resources\ProjectResource;

class ContentService implements ContentServiceInterface
{
    public function __construct(
        protected ContentRepository $contents,
    ) {}

    public function listByType(array $filters = [], int $perPage = 15, int $page = 1, string $type): LengthAwarePaginator
    {
        return $this->contents->paginate($filters, $perPage, $page, $type);
    }

    public function listAll(array $filters = [], int $perPage = 15, int $page = 1): LengthAwarePaginator
    {
        return $this->contents->paginate($filters, $perPage, $page, '');
    }

    public function get(string $slug): Content
    {
        return $this->contents->findBySlug($slug);
    }

    public function getByType(string $slug, string $type): Content
    {
        return $this->contents->findBySlugAndType($slug, $type);
    }

    public function create(array $data): Content
    {
        return $this->contents->create($data);
    }

    public function update(string $id, array $data): Content
    {
        $content = $this->contents->findById($id);

        return $this->contents->update($content, $data);
    }

    public function delete(string $id): void
    {
        $content = $this->contents->findById($id);

        $this->contents->delete($content);
    }

    public function listContentTypes(array $filters = []): array
    {
        if (isset($filters['category'])) {
            return match ($filters['category']) {
                'knowledge_base' => [
                    DictionaryHelper::getDictionaryBySlug(DictionaryEntryResource::getDictionarySlug()),
                    DictionaryHelper::getDictionaryBySlug(JobPositionDescriptionResource::getDictionarySlug()),
                    DictionaryHelper::getDictionaryBySlug(QualificationResource::getDictionarySlug()),
                    DictionaryHelper::getDictionaryBySlug(TechnologyToolResource::getDictionarySlug()),
                ],
                'articles' => [
                    DictionaryHelper::getDictionaryBySlug(ArticleResource::getDictionarySlug()),
                ],
                'projects' => [
                    DictionaryHelper::getDictionaryBySlug(ProjectResource::getDictionarySlug()),
                ],
                default => [
                    DictionaryHelper::getDictionaryBySlug(DictionaryEntryResource::getDictionarySlug()),
                    DictionaryHelper::getDictionaryBySlug(JobPositionDescriptionResource::getDictionarySlug()),
                    DictionaryHelper::getDictionaryBySlug(QualificationResource::getDictionarySlug()),
                    DictionaryHelper::getDictionaryBySlug(TechnologyToolResource::getDictionarySlug()),
                    DictionaryHelper::getDictionaryBySlug(ArticleResource::getDictionarySlug()),
                    DictionaryHelper::getDictionaryBySlug(ProjectResource::getDictionarySlug()),
                ],
            };
        }

        return [
            DictionaryHelper::getDictionaryBySlug(DictionaryEntryResource::getDictionarySlug()),
            DictionaryHelper::getDictionaryBySlug(JobPositionDescriptionResource::getDictionarySlug()),
            DictionaryHelper::getDictionaryBySlug(QualificationResource::getDictionarySlug()),
            DictionaryHelper::getDictionaryBySlug(TechnologyToolResource::getDictionarySlug()),
            DictionaryHelper::getDictionaryBySlug(ArticleResource::getDictionarySlug()),
            DictionaryHelper::getDictionaryBySlug(ProjectResource::getDictionarySlug()),
        ];
    }

    public function determineSeoKey(?string $contentCategoryIds): string
    {
        if (!$contentCategoryIds) {
            return 'all';
        }

        if (str_contains($contentCategoryIds, ',')) {
            return 'knowledge-base';
        }

        $slug = DictionaryHelper::getSlugById($contentCategoryIds);

        return $slug ?: 'all';
    }

    public function getListForSitemap(string $contentCategoryId, int $perPage = 50, int $page = 1): LengthAwarePaginator
    {
        return $this->contents->getListForSitemap($contentCategoryId, $perPage, $page);
    }

    public function getDictionaryFiltersForContent(array $filters, int $perPage = 15): LengthAwarePaginator
    {
        return $this->contents->getDictionaryFiltersForContent($filters, $perPage);

    }
}
