<?php

namespace MbeGroup\Content\Repository;

use MbeGroup\Content\Models\Content;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Collection;
use App\ValueObject\Status;
use App\Models\Dictionary;

class ContentRepository
{
    public function paginate(array $filters = [], int $perPage = 15, int $page = 1, string $type): LengthAwarePaginator
    {
        $query = Content::query()
            ->when($type !== '', function ($query) use ($type) {
                $query->where('type', $type);
            })
            ->when($filters['q'] ?? null, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('title', 'like', "%{$search}%");
                });
            })
            ->when($filters['category_ids'] ?? null, function ($query, $categoryIds) {
                $query->whereIn('content_category_id', array_map('intval', explode(',', $categoryIds)));
            })
            ->when($filters['industry_ids'] ?? null, function ($query, $industryIds) {
                $query->whereIn('industry_ids', array_map('intval', explode(',', $industryIds)));
            })
            ->when($filters['specialization_ids'] ?? null, function ($query, $specializationIds) {
                $query->whereIn('specialization_id', array_map('intval', explode(',', $specializationIds)));
            })
            ->when($filters['employer_id'] ?? null, function ($query, $employerId) {
                $query->where('employer_id', $employerId);
            })
            ->when($filters['author_ids'] ?? null, function ($query, $authorIds) {
                $query->whereIn('author_ids', array_map('strval', explode(',', $authorIds)));
            })
            ->when($filters['content_category_ids'] ?? null, function ($query, $contentCategoryIds) {
                $query->whereIn('content_category_id', array_map('intval', explode(',', $contentCategoryIds)));
            })
            ->where('status', Status::ACTIVE)
            ->where(function ($query) {
                $query->whereNull('publication_date')
                    ->orWhere('publication_date', '<=', now());
            })
            ->orderByDesc('created_at');

        return $query->paginate($perPage, ['*'], 'page', $page);
    }

    public function all(): Collection
    {
        return Content::query()
            ->orderByDesc('created_at')
            ->where('status', Status::ACTIVE)
            ->where('publication_date', '<=', now())
            ->get();
    }

    public function findById(string $id): Content
    {
        return Content::where('id', $id)
            ->where('status', Status::ACTIVE)
            ->where('publication_date', '<=', now())
            ->firstOrFail();
    }

    public function findBySlug(string $slug): Content
    {
        return Content::where('slug', $slug)
            ->where('status', Status::ACTIVE)
            ->where('publication_date', '<=', now())
            ->firstOrFail();
    }

    public function findBySlugAndType(string $slug, string $type): Content
    {
        return Content::where('slug', $slug)
            ->where('type', $type)
            ->where('status', Status::ACTIVE)
            ->where('publication_date', '<=', now())
            ->firstOrFail();
    }

    public function create(array $data): Content
    {
        return Content::create($data);
    }

    public function update(Content $content, array $data): Content
    {
        $content->update($data);

        return $content;
    }

    public function delete(Content $content): void
    {
        $content->delete();
    }

    public function getListForSitemap(string $contentCategoryId, int $perPage = 50, int $page = 1): LengthAwarePaginator
    {
        $query = Content::query()
            ->where('status', Status::ACTIVE)
            ->where(function ($query) {
                $query->whereNull('publication_date')
                    ->orWhere('publication_date', '<=', now());
            })
            ->select([
                'id',
                'slug',
                'title',
                'updated_at',
                'created_at',
                'content_category_id',
            ]);

        if ($contentCategoryId) {
            $query->where('content_category_id', (int) $contentCategoryId);
        }

        return $query
            ->orderBy('updated_at', 'desc')
            ->paginate($perPage, ['*'], 'page', $page);
    }

    public function getDictionaryFiltersForContent(array $filters = [], int $perPage = 15): LengthAwarePaginator
    {
        return Dictionary::query()
            ->when($filters['type'] ?? null, function ($query, $type) {
                $query->where('type', $type);
            })
            ->when($filters['in_content'] ?? false, function ($query) use ($filters) {
                $contentCategoryIds = $filters['content_category_ids'] ?? null;

                if (is_string($contentCategoryIds)) {
                    $contentCategoryIds = array_map('intval', explode(',', $contentCategoryIds));
                }

                $usedDictionaryIds = $this->getUsedDictionaryIdsFromMongo($contentCategoryIds);

                if (!empty($usedDictionaryIds)) {
                    $query->whereIn('id', $usedDictionaryIds);
                } else {
                    $query->whereRaw('1 = 0');
                }
            })
            ->orderByDesc('created_at')
            ->paginate($perPage);
    }

    private function getUsedDictionaryIdsFromMongo(?array $contentCategoryIds = null): array
    {
        $collection = Content::raw();

        $matchConditions = [
            'status' => Status::ACTIVE
        ];

        if (!empty($contentCategoryIds)) {
            $matchConditions['content_category_id'] = ['$in' => $contentCategoryIds];
        }

        $pipeline = [
            [
                '$match' => $matchConditions
            ],
            [
                '$project' => [
                    'all_dictionary_ids' => [
                        '$concatArrays' => [
                            ['$cond' => ['if' => ['$ne' => ['$content_category_id', null]], 'then' => ['$content_category_id'], 'else' => []]],
                            ['$cond' => ['if' => ['$ne' => ['$project_type', null]], 'then' => ['$project_type'], 'else' => []]],
                            ['$cond' => ['if' => ['$ne' => ['$project_scope', null]], 'then' => ['$project_scope'], 'else' => []]],
                            ['$ifNull' => ['$industry_ids', []]],
                            ['$ifNull' => ['$specialization_ids', []]],
                            ['$ifNull' => ['$team_composition', []]],
                            ['$ifNull' => ['$software', []]],
                            ['$ifNull' => ['$hardware', []]],
                            ['$ifNull' => ['$methodologies', []]],
                        ]
                    ]
                ]
            ],
            [
                '$unwind' => '$all_dictionary_ids'
            ],
            [
                '$group' => [
                    '_id' => '$all_dictionary_ids'
                ]
            ]
        ];

        $results = $collection->aggregate($pipeline)->toArray();

        return array_map(fn($item) => $item['_id'], $results);
    }
}
