<?php
namespace MbeGroup\Content\Models;

use MongoDB\Laravel\Eloquent\Model;
use App\ValueObject\Status;
use App\Models\Dictionary;
use Illuminate\Database\Eloquent\Builder;
use MbeGroup\Employer\Models\Employer;
use App\Traits\HasSeoFields;
use MbeGroup\Author\Models\Author;
use App\Traits\HasDictionaryRelations;
use MongoDB\BSON\ObjectId;
use Illuminate\Database\Eloquent\Casts\Attribute;
use App\Filament\Forms\Components\RichEditor\RichContentCustomBlocks;

class Content extends Model
{
    use HasSeoFields, HasDictionaryRelations;

    protected $connection = 'mongodb';
    protected $collection = 'contents';
    protected $fillable   = [
        'title',
        'subtitle',
        'slug',
        'short_title',
        'content_category_id',
        'industry_ids',
        'specialization_ids',
        'employer_id',
        'location',
        'is_sponsored',
        'is_sticky',
        'main_image',
        'main_image_caption',
        'status',
        'type',
        'lead',
        'content',
        'tags',
        'publication_date',
        'author_ids',
        'expert_ids',
        'employer_image',
        'project_type',
        'project_scope',
        'project_duration',
        'people_involved_count',
        'key_numbers',
        'gallery',
        'team_description',
        'involved_people',
        'team_composition',
        'software',
        'hardware',
        'methodologies',
    ];

    protected $casts = [
        'status'             => Status::class,
        'gallery'            => 'array',
        'publication_date'   => 'datetime',
    ];

    protected function renderedContent(): Attribute
    {
        return Attribute::make(
            get: fn () => $this->processCustomBlocks($this->content)
        );
    }

    private function processCustomBlocks($html): string
    {
        return preg_replace_callback(
            '/<div\s+data-type="customBlock"\s+data-config="([^"]+)"\s+data-id="([^"]+)"><\/div>/i',
            function ($matches) {
                $config = html_entity_decode($matches[1]);
                $blockId = $matches[2];

                $configArray = json_decode($config, true) ?? [];

                return match ($blockId) {
                    RichContentCustomBlocks\ExpertFrameBlock::getId() => RichContentCustomBlocks\ExpertFrameBlock::toHtml([], $configArray),
                    RichContentCustomBlocks\EmbedFrameBlock::getId() => RichContentCustomBlocks\EmbedFrameBlock::toHtml([], $configArray),
                    RichContentCustomBlocks\IFrameBlock::getId() => RichContentCustomBlocks\IFrameBlock::toHtml([], $configArray),
                    RichContentCustomBlocks\SchematFrameBlock::getId() => RichContentCustomBlocks\SchematFrameBlock::toHtml([], $configArray),
                    default => $matches[0],
                };
            },
            $html
        );
    }

    public function contentCategory()
    {
        return $this->belongsTo(Dictionary::class, 'content_category_id');
    }

    public function employer()
    {
        return $this->belongsTo(Employer::class, 'employer_id');
    }

    public function getSpecializations()
    {
        return $this->getDictionaryItems($this->specialization_ids ?? null);
    }

    public function getIndustries()
    {
        return $this->getDictionaryItems($this->industry_ids ?? null);
    }

    public function getCategory()
    {
        return $this->getDictionaryItemsWithParents($this->content_category_id ? [$this->content_category_id] : null);
    }

    public function scopeOfType(Builder $query, string $type): Builder
    {
        return $query->where('type', $type);
    }

    public function experts()
    {
        $ids = array_map(fn($id) => new ObjectId($id), $this->expert_ids ?? []);
        return Author::query()->whereIn('_id', $ids);
    }

    public function getAuthorsListAttribute()
    {
        $ids = array_map(fn($id) => new ObjectId($id), $this->author_ids ?? []);
        return \MbeGroup\Author\Models\Author::whereIn('_id', $ids)->get();
    }

    public function getMainImageUrlAttribute()
    {
        return $this->main_image ? url('storage/' . ltrim($this->main_image, '/')) : null;
    }

    public function getGalleryImageUrlsAttribute()
    {
        if (empty($this->gallery) || !is_array($this->gallery)) {
            return null;
        }

        return array_map(function ($item) {
            if (is_array($item) && isset($item['image'])) {
                return url('storage/' . ltrim($item['image'], '/'));
            }
            return null;
        }, $this->gallery);
    }

    public function getInvolvedPersonImageUrlAttribute($person)
    {
        return isset($person['image']) && $person['image']
            ? url('storage/' . ltrim($person['image'], '/'))
            : null;
    }

    public function getTeamComposition()
    {
        return $this->getDictionaryItems($this->team_composition ?? null);
    }

    public function getSoftware()
    {
        return $this->getDictionaryItems($this->software ?? null);
    }

    public function getHardware()
    {
        return $this->getDictionaryItems($this->hardware ?? null);
    }

    public function getMethodologies()
    {
        return $this->getDictionaryItems($this->methodologies ?? null);
    }

    public function getProjectType()
    {
        return $this->getDictionaryItems($this->project_type ? [$this->project_type] : null);
    }

    public function getProjectScope()
    {
        return $this->getDictionaryItems($this->project_scope ? [$this->project_scope] : null);
    }
}
