<?php

namespace MbeGroup\Content\Filament\Resources;

use MbeGroup\Content\Models\Content;
use Filament\Schemas\Schema;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Filament\Actions;
use App\ValueObject\Status;
use Filament\Schemas\Components\Section;
use Filament\Forms;
use App\Helpers\DictionaryHelper;
use MbeGroup\Author\Models\Author;
use Tapp\FilamentGoogleAutocomplete\Forms\Components\GoogleAutocomplete;
use App\Filament\Components\SeoFormComponents;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use App\Filament\Components\RichEditorFormComponents;

abstract class BaseContentResource extends Resource
{
    protected static ?string $model = Content::class;

    protected static string $contentType;

    public static function getNavigationGroup(): ?string
    {
        return __('content::filament.navigation.group.label');
    }

    public static function form(Schema $schema): Schema
    {
        $statusOptions = Status::getLabels();

        $dictionaryOptions = [
            'industry'         => DictionaryHelper::getHierarchicalNamesByType('industry'),
            'specialization'   => DictionaryHelper::getHierarchicalNamesByType('specialization'),
            'content_category' => DictionaryHelper::getHierarchicalNamesByType('content_category'),
            'author'           => DictionaryHelper::getIdBySlug('author'),
            'expert'           => DictionaryHelper::getIdBySlug('expert'),
        ];

        return parent::form($schema)
            ->schema([
                Forms\Components\Hidden::make('type')
                    ->default(static::$contentType),
                Tabs::make('Tabs')
                ->persistTabInQueryString()
                ->tabs([
                    Tab::make(__('content::filament.resources.article.form.tabs.basic_info.label'))
                        ->schema([
                            Forms\Components\TextInput::make('title')
                                ->label(__('content::filament.resources.article.form.fields.title'))
                                ->required()
                                ->maxLength(255)
                                ->live(onBlur: true)
                                ->afterStateUpdated(function (string $operation, $state, callable $set) {
                                    if ($operation !== 'create') {
                                        return;
                                    }
                                    $set('slug', \Illuminate\Support\Str::slug($state));
                                }),
                            Forms\Components\TextInput::make('slug')
                                ->label(__('content::filament.resources.article.form.fields.slug'))
                                ->required()
                                ->maxLength(255)
                                ->disabled(fn (string $operation): bool => $operation === 'edit')
                                ->dehydrated(fn (string $operation): bool => $operation === 'create'),
                            Forms\Components\TextInput::make('short_title')
                                ->label(__('content::filament.resources.article.form.fields.short_title'))
                                ->maxLength(255),
                            Forms\Components\Select::make('author_ids')
                                ->label(__('content::filament.resources.article.form.fields.author_ids'))
                                ->options(Author::getSelectOptions($dictionaryOptions['author'] ?? null))
                                ->searchable()
                                ->allowHtml()
                                ->multiple()
                                ->getSearchResultsUsing(fn (string $search) => Author::searchForSelect($search, $dictionaryOptions['author'] ?? null)),
                            Forms\Components\Select::make('expert_ids')
                                ->label(__('content::filament.resources.article.form.fields.expert_ids'))
                                ->options(Author::getSelectOptions($dictionaryOptions['expert'] ?? null))
                                ->searchable()
                                ->allowHtml()
                                ->multiple()
                                ->getSearchResultsUsing(fn (string $search) => Author::searchForSelect($search, $dictionaryOptions['expert'] ?? null)),
                            Forms\Components\Select::make('content_category_id')
                                ->label(__('content::filament.resources.article.form.fields.content_category_id'))
                                ->options($dictionaryOptions['content_category'])
                                ->allowHtml()
                                ->searchable()
                                ->default(function () {
                                    $slug = basename(static::$slug);
                                    $defaultContentCategoryId = DictionaryHelper::getIdBySlug($slug);

                                    return $defaultContentCategoryId ?? null;
                                }),
                            Forms\Components\Select::make('industry_ids')
                                ->label(__('content::filament.resources.article.form.fields.industry_ids'))
                                ->options($dictionaryOptions['industry'])
                                ->allowHtml()
                                ->searchable()
                                ->multiple(),
                            Forms\Components\Select::make('specialization_ids')
                                ->label(__('content::filament.resources.article.form.fields.specialization_ids'))
                                ->label(__('content::filament.resources.article.form.fields.specialization_ids'))
                                ->options($dictionaryOptions['specialization'])
                                ->allowHtml()
                                ->searchable()
                                ->multiple(),
                            Forms\Components\Select::make('employer_id')
                                ->label(__('content::filament.resources.article.form.fields.employer_id'))
                                ->options(\MbeGroup\Employer\Models\Employer::getSelectOptions())
                                ->searchable()
                                ->getSearchResultsUsing(fn (string $search) => \MbeGroup\Employer\Models\Employer::searchForSelect($search)),
                            Forms\Components\DateTimePicker::make('publication_date')
                                ->label(__('content::filament.resources.article.form.fields.publication_date'))
                                ->default(now())
                                ->required(),
                            Section::make(__('content::filament.resources.article.form.fields.location'))
                                ->collapsible()
                                ->collapsed()
                                ->schema([
                                    Forms\Components\Hidden::make('location')
                                        ->dehydrateStateUsing(fn ($state, $get) => [
                                            'location_name' => $get('location_name'),
                                            'municipality' => $get('municipality'),
                                            'region' => $get('region'),
                                            'postal_code' => $get('postal_code'),
                                            'street' => $get('street'),
                                            'building_number' => $get('building_number'),
                                            'latitude' => $get('latitude'),
                                            'longitude' => $get('longitude'),
                                        ])
                                        ->afterStateHydrated(function (Forms\Components\Hidden $component, $state, callable $set): void {
                                            if (is_array($state)) {
                                                $set('location_name', $state['location_name'] ?? '');
                                                $set('municipality', $state['municipality'] ?? '');
                                                $set('region', $state['region'] ?? '');
                                                $set('postal_code', $state['postal_code'] ?? '');
                                                $set('street', $state['street'] ?? '');
                                                $set('building_number', $state['building_number'] ?? '');
                                                $set('latitude', $state['latitude'] ?? '');
                                                $set('longitude', $state['longitude'] ?? '');
                                            }
                                        }),
                                    GoogleAutocomplete::make('google_search')
                                        ->label(__('employer::filament.resources.employer.form.fields.google_search'))
                                        ->countries([
                                            'PL',
                                        ])
                                        ->autocompletePlaceholder('Wyszukaj lokalizację')
                                        ->language('pl')
                                        ->lazy()
                                        ->dehydrated(false)
                                        ->withFields([
                                            Forms\Components\TextInput::make('location_name')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{locality}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.location_name')),
                                            Forms\Components\TextInput::make('municipality')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{administrative_area_level_2}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.municipality')),
                                            Forms\Components\TextInput::make('region')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{administrative_area_level_1}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.region')),
                                            Forms\Components\TextInput::make('postal_code')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{postal_code}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.postal_code')),
                                            Forms\Components\TextInput::make('street')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{route}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.street')),
                                            Forms\Components\TextInput::make('building_number')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{street_number}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.building_number')),
                                            Forms\Components\TextInput::make('latitude')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{latitude}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.latitude')),
                                            Forms\Components\TextInput::make('longitude')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{longitude}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.longitude')),
                                        ]),
                                ])->columns(2),
                            Forms\Components\Toggle::make('is_sponsored')
                                ->label(__('content::filament.resources.article.form.fields.is_sponsored')),
                            Forms\Components\Toggle::make('is_sticky')
                                ->label(__('content::filament.resources.article.form.fields.is_sticky'))
                                ->default(true),
                            Forms\Components\FileUpload::make('main_image')
                                ->label(__('content::filament.resources.article.form.fields.main_image'))
                                ->image()
                                ->directory(static::$contentType . '/main_images')
                                ->disk('public')
                                ->downloadable()
                                ->openable()
                                ->maxSize(2048),
                            Forms\Components\TextInput::make('main_image_caption')
                                ->label(__('content::filament.resources.article.form.fields.main_image_caption'))
                                ->maxLength(255),
                            Forms\Components\Select::make('status')
                                ->label(__('content::filament.resources.article.form.fields.status'))
                                ->options($statusOptions)
                                ->required()
                                ->default(Status::INACTIVE)
                                ->afterStateHydrated(function (Forms\Components\Select $component, $state): void {
                                    if ($state instanceof Status) {
                                        $component->state($state->value);
                                    }
                                })
                                ->dehydrateStateUsing(function ($state): int {
                                    return $state instanceof Status ? $state->value: (int) $state;
                                })
                                ->rule('integer'),
                        ]),
                    Tab::make(__('content::filament.resources.article.form.tabs.content.label'))
                        ->schema([
                            Forms\Components\Textarea::make('lead')
                                ->label(__('content::filament.resources.article.form.fields.lead'))
                                ->required()
                                ->rows(5),
                            RichEditorFormComponents::getRichEditor(
                                __('content::filament.resources.article.form.fields.content'),
                                'content'
                            ),
                            Forms\Components\TagsInput::make('tags')
                                ->label(__('content::filament.resources.article.form.fields.tags'))
                                ->placeholder(__('content::filament.resources.article.form.fields.tags_placeholder'))
                        ]),
                    Tab::make(__('content::filament.resources.article.form.tabs.gallery.label'))
                        ->schema([
                            Forms\Components\Repeater::make('gallery')
                                ->label(__('content::filament.resources.project.form.fields.gallery'))
                                ->schema([
                                    Forms\Components\FileUpload::make('image')
                                        ->label(__('content::filament.resources.project.form.fields.image'))
                                        ->image()
                                        ->directory(static::$contentType . '/gallery')
                                        ->disk('public')
                                        ->downloadable()
                                        ->openable()
                                        ->required()
                                        ->maxSize(4096)
                                        ->imageEditor()
                                        ->imagePreviewHeight('150')
                                        ->columnSpan('full'),
                                    Forms\Components\TextInput::make('image_caption')
                                        ->label(__('content::filament.resources.project.form.fields.image_caption'))
                                        ->maxLength(255)
                                        ->placeholder(__('content::filament.resources.project.form.fields.image_caption_placeholder'))
                                        ->columnSpan('full'),
                                ])
                                ->default([])
                                ->columns(1)
                                ->grid(3)
                                ->addActionLabel(__('content::filament.resources.project.form.fields.gallery_create_button_label'))
                                ->minItems(0)
                                ->maxItems(50)
                                ->columnSpan('full')
                                ->collapsible()
                                ->collapsed(false)
                                ->reorderable()
                                ->itemLabel(fn (array $state): ?string => $state['image_caption'] ?? __('content::filament.resources.project.form.fields.gallery_item_label_default'))
                                ->deleteAction(
                                    fn (Actions\Action $action) => $action->requiresConfirmation()
                                ),
                        ]
                    ),
                    SeoFormComponents::getSeoTab(),
                ]),
            ])->columns(1);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label(__('content::filament.resources.content.table.columns.id'))
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->sortable(),
                Tables\Columns\TextColumn::make('title')
                    ->label(__('content::filament.resources.content.table.columns.title'))
                    ->searchable()
                    ->sortable()
                    ->limit(40),
                Tables\Columns\TextColumn::make('contentCategory.name')
                    ->label(__('content::filament.resources.content.table.columns.category'))
                    ->placeholder('Brak danych')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('employer.name')
                    ->label(__('content::filament.resources.content.table.columns.employer'))
                    ->placeholder('Brak danych')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\IconColumn::make('status')
                    ->label(__('content::filament.resources.content.table.columns.status'))
                    ->state(fn ($record) => Status::int($record->status))
                    ->icon(fn ($state) => Status::icon($state))
                    ->color(fn ($state) => Status::color($state))
                    ->tooltip(fn ($state) => Status::label($state))
                    ->sortable(),
                Tables\Columns\TextColumn::make('publication_date')
                    ->label(__('content::filament.resources.content.table.columns.publication_date'))
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label(__('content::filament.resources.content.table.columns.created_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->label(__('content::filament.resources.content.table.columns.updated_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->recordActions([
                Actions\EditAction::make(),
                Actions\DeleteAction::make(),
            ])
            ->toolbarActions([
                Actions\BulkActionGroup::make([
                    Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->where('type', static::$contentType)
            ->with(['employer', 'contentCategory']);
    }

    protected static function getTypeSpecificFields(): array
    {
        return [];
    }

    public static function getContentType(): string
    {
        return static::$contentType;
    }
}
