<?php

namespace MbeGroup\Content\Services;

use MbeGroup\Content\Repository\ContentRepository;
use MbeGroup\Content\Models\Content;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use MbeGroup\Content\Contracts\ContentServiceInterface;
use MbeGroup\Content\Filament\Resources\ArticleResource;
use MbeGroup\Content\Filament\Clusters\KnowledgeBase\Resources\QualificationResource;
use MbeGroup\Content\Filament\Clusters\KnowledgeBase\Resources\JobPositionDescriptionResource;
use MbeGroup\Content\Filament\Clusters\KnowledgeBase\Resources\TechnologyToolResource;
use MbeGroup\Content\Filament\Clusters\KnowledgeBase\Resources\DictionaryEntryResource;

class ContentService implements ContentServiceInterface
{
    public function __construct(
        protected ContentRepository $contents,
    ) {}

    public function listByType(array $filters = [], int $perPage = 15, int $page = 1, string $type): LengthAwarePaginator
    {
        return $this->contents->paginate($filters, $perPage, $page, $type);
    }

    public function listAll(array $filters = [], int $perPage = 15, int $page = 1): LengthAwarePaginator
    {
        return $this->contents->paginate($filters, $perPage, $page, '');
    }

    public function get(string $slug): Content
    {
        return $this->contents->findBySlug($slug);
    }

    public function getByType(string $slug, string $type): Content
    {
        return $this->contents->findBySlugAndType($slug, $type);
    }

    public function create(array $data): Content
    {
        return $this->contents->create($data);
    }

    public function update(string $id, array $data): Content
    {
        $content = $this->contents->findById($id);

        return $this->contents->update($content, $data);
    }

    public function delete(string $id): void
    {
        $content = $this->contents->findById($id);

        $this->contents->delete($content);
    }

    public function listContentTypes(array $filters = []): array
    {
        if (isset($filters['category'])) {
            return match ($filters['category']) {
                'knowledge_base' => [
                    QualificationResource::getContentType()          => __('content::filament.resources.qualifications.navigation_label'),
                    JobPositionDescriptionResource::getContentType() => __('content::filament.resources.job_position_descriptions.navigation_label'),
                    TechnologyToolResource::getContentType()         => __('content::filament.resources.technology_tools.navigation_label'),
                    DictionaryEntryResource::getContentType()        => __('content::filament.resources.dictionary_terms.navigation_label'),
                ],
                'articles' => [
                    ArticleResource::getContentType() => __('content::filament.resources.article.navigation_label'),
                ],
                default => [
                    DictionaryEntryResource::getContentType()        => __('content::filament.resources.dictionary_terms.navigation_label'),
                    QualificationResource::getContentType()          => __('content::filament.resources.qualifications.navigation_label'),
                    JobPositionDescriptionResource::getContentType() => __('content::filament.resources.job_position_descriptions.navigation_label'),
                    ArticleResource::getContentType()                => __('content::filament.resources.article.navigation_label'),
                    TechnologyToolResource::getContentType()         => __('content::filament.resources.technology_tools.navigation_label'),
                ],
            };
        }

        return [
            DictionaryEntryResource::getContentType()        => __('content::filament.resources.dictionary_terms.navigation_label'),
            QualificationResource::getContentType()          => __('content::filament.resources.qualifications.navigation_label'),
            JobPositionDescriptionResource::getContentType() => __('content::filament.resources.job_position_descriptions.navigation_label'),
            ArticleResource::getContentType()                => __('content::filament.resources.article.navigation_label'),
            TechnologyToolResource::getContentType()         => __('content::filament.resources.technology_tools.navigation_label'),
        ];
    }
}
