<?php

namespace MbeGroup\Content\Filament\Resources;

use MbeGroup\Content\Filament\Resources\BaseContentResource;
use Filament\Schemas\Schema;
use Filament\Actions\Action;
use App\ValueObject\Status;
use Filament\Schemas\Components\Section;
use Filament\Forms;
use App\Helpers\DictionaryHelper;
use Tapp\FilamentGoogleAutocomplete\Forms\Components\GoogleAutocomplete;
use App\Filament\Components\SeoFormComponents;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use App\Filament\Components\RichEditorFormComponents;

class ProjectResource extends BaseContentResource
{
    protected static string $contentType = 'project';

    protected static ?string $slug = 'content/projects';

    protected static ?string $dictionarySlug = 'projects';

    public static function getNavigationGroup(): ?string
    {
        return __('content::filament.navigation.group.label');
    }

    public static function getNavigationSort(): ?int
    {
        return 2;
    }

    public static function getNavigationLabel(): string
    {
        return __('content::filament.resources.project.navigation_label');
    }

    public static function getModelLabel(): string
    {
        return __('content::filament.resources.project.model_label');
    }

    public static function getPluralModelLabel(): string
    {
        return __('content::filament.resources.project.plural_model_label');
    }

    public static function form(Schema $schema): Schema
    {
        $statusOptions = Status::getLabels();

        $dictionaryOptions = [
            'industry'         => DictionaryHelper::getHierarchicalNamesByType('industry'),
            'specialization'   => DictionaryHelper::getHierarchicalNamesByType('specialization'),
            'content_category' => DictionaryHelper::getHierarchicalNamesByType('content_category'),
            'author'           => DictionaryHelper::getIdBySlug('author'),
            'expert'           => DictionaryHelper::getIdBySlug('expert'),
            'project_type'     => DictionaryHelper::getHierarchicalNamesByType('project_type'),
            'project_scope'    => DictionaryHelper::getHierarchicalNamesByType('project_scope'),
            'position'         => DictionaryHelper::getHierarchicalNamesByType('position'),
            'software'         => DictionaryHelper::getHierarchicalNamesByType('software'),
            'hardware'         => DictionaryHelper::getHierarchicalNamesByType('hardware'),
            'methodologies'    => DictionaryHelper::getHierarchicalNamesByType('methodologies'),
        ];

        return parent::form($schema)
            ->schema([
                Forms\Components\Hidden::make('type')
                    ->default(static::$contentType),
                Tabs::make('Tabs')
                ->persistTabInQueryString()
                ->tabs([
                    Tab::make(__('content::filament.resources.article.form.tabs.basic_info.label'))
                        ->schema([
                            Forms\Components\TextInput::make('title')
                                ->label(__('content::filament.resources.article.form.fields.title'))
                                ->required()
                                ->maxLength(255)
                                ->live(onBlur: true)
                                ->afterStateUpdated(function (string $operation, $state, callable $set) {
                                    if ($operation !== 'create') {
                                        return;
                                    }
                                    $set('slug', \Illuminate\Support\Str::slug($state));
                                }),
                            Forms\Components\TextInput::make('slug')
                                ->label(__('content::filament.resources.article.form.fields.slug'))
                                ->required()
                                ->maxLength(255)
                                ->disabled(fn (string $operation): bool => $operation === 'edit')
                                ->dehydrated(fn (string $operation): bool => $operation === 'create'),
                            Forms\Components\TextInput::make('subtitle')
                                ->label(__('content::filament.resources.article.form.fields.subtitle'))
                                ->maxLength(255),
                            Forms\Components\Select::make('employer_id')
                                ->label(__('content::filament.resources.article.form.fields.employer_id'))
                                ->options(\MbeGroup\Employer\Models\Employer::getSelectOptions())
                                ->searchable()
                                ->required()
                                ->getSearchResultsUsing(fn (string $search) => \MbeGroup\Employer\Models\Employer::searchForSelect($search)),
                            Forms\Components\FileUpload::make('employer_image')
                                ->label(__('content::filament.resources.project.form.fields.employer_image'))
                                ->image()
                                ->directory(static::$contentType . '/employer_images')
                                ->disk('public')
                                ->downloadable()
                                ->openable()
                                ->maxSize(2048),
                            Forms\Components\DateTimePicker::make('publication_date')
                                ->label(__('content::filament.resources.article.form.fields.publication_date'))
                                ->default(now())
                                ->required(),
                            Forms\Components\Select::make('content_category_id')
                                ->label(__('content::filament.resources.article.form.fields.content_category_id'))
                                ->options($dictionaryOptions['content_category'])
                                ->allowHtml()
                                ->searchable()
                                ->default(function () {
                                    $slug = basename(self::$slug);
                                    $defaultContentCategoryId = DictionaryHelper::getIdBySlug($slug);

                                    return $defaultContentCategoryId ?? null;
                                }),
                            Forms\Components\Select::make('industry_ids')
                                ->label(__('content::filament.resources.article.form.fields.industry_ids'))
                                ->options($dictionaryOptions['industry'])
                                ->allowHtml()
                                ->searchable()
                                ->multiple(),
                            Forms\Components\Select::make('specialization_ids')
                                ->label(__('content::filament.resources.article.form.fields.specialization_ids'))
                                ->label(__('content::filament.resources.article.form.fields.specialization_ids'))
                                ->options($dictionaryOptions['specialization'])
                                ->allowHtml()
                                ->searchable()
                                ->multiple(),
                            Section::make(__('content::filament.resources.article.form.fields.location'))
                                ->collapsible()
                                ->collapsed()
                                ->schema([
                                    Forms\Components\Hidden::make('location')
                                        ->dehydrateStateUsing(fn ($state, $get) => [
                                            'location_name'   => $get('location_name'),
                                            'municipality'    => $get('municipality'),
                                            'region'          => $get('region'),
                                            'postal_code'     => $get('postal_code'),
                                            'street'          => $get('street'),
                                            'building_number' => $get('building_number'),
                                            'latitude'        => $get('latitude'),
                                            'longitude'       => $get('longitude'),
                                        ])
                                        ->afterStateHydrated(function (Forms\Components\Hidden $component, $state, callable $set): void {
                                            if (is_array($state)) {
                                                $set('location_name', $state['location_name'] ?? '');
                                                $set('municipality', $state['municipality'] ?? '');
                                                $set('region', $state['region'] ?? '');
                                                $set('postal_code', $state['postal_code'] ?? '');
                                                $set('street', $state['street'] ?? '');
                                                $set('building_number', $state['building_number'] ?? '');
                                                $set('latitude', $state['latitude'] ?? '');
                                                $set('longitude', $state['longitude'] ?? '');
                                            }
                                        }),
                                    GoogleAutocomplete::make('google_search')
                                        ->label(__('employer::filament.resources.employer.form.fields.google_search'))
                                        ->countries([
                                            'PL',
                                        ])
                                        ->autocompletePlaceholder('Wyszukaj lokalizację')
                                        ->language('pl')
                                        ->lazy()
                                        ->dehydrated(false)
                                        ->withFields([
                                            Forms\Components\TextInput::make('location_name')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{locality}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.location_name')),
                                            Forms\Components\TextInput::make('municipality')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{administrative_area_level_2}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.municipality')),
                                            Forms\Components\TextInput::make('region')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{administrative_area_level_1}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.region')),
                                            Forms\Components\TextInput::make('postal_code')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{postal_code}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.postal_code')),
                                            Forms\Components\TextInput::make('street')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{route}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.street')),
                                            Forms\Components\TextInput::make('building_number')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{street_number}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.building_number')),
                                            Forms\Components\TextInput::make('latitude')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{latitude}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.latitude')),
                                            Forms\Components\TextInput::make('longitude')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{longitude}',
                                                ])
                                                ->label(__('employer::filament.resources.employer.form.fields.longitude')),
                                        ]),
                                ])->columns(2),
                            Forms\Components\Select::make('project_type')
                                ->label(__('content::filament.resources.project.form.fields.project_type'))
                                ->options($dictionaryOptions['project_type'])
                                ->allowHtml()
                                ->searchable(),
                            Forms\Components\Select::make('project_scope')
                                ->label(__('content::filament.resources.project.form.fields.project_scope'))
                                ->options($dictionaryOptions['project_scope'])
                                ->allowHtml()
                                ->searchable(),
                            Forms\Components\TextInput::make('people_involved_count')
                                ->label(__('content::filament.resources.project.form.fields.people_involved'))
                                ->maxLength(255),
                            Forms\Components\TextInput::make('project_duration')
                                ->label(__('content::filament.resources.project.form.fields.project_duration'))
                                ->maxLength(255),
                            Forms\Components\FileUpload::make('main_image')
                                ->label(__('content::filament.resources.article.form.fields.main_image'))
                                ->image()
                                ->directory(static::$contentType . '/main_images')
                                ->disk('public')
                                ->downloadable()
                                ->openable()
                                ->maxSize(4096),
                            Forms\Components\TextInput::make('main_image_caption')
                                ->label(__('content::filament.resources.article.form.fields.main_image_caption'))
                                ->maxLength(255),
                            Forms\Components\Select::make('status')
                                ->label(__('content::filament.resources.article.form.fields.status'))
                                ->options($statusOptions)
                                ->required()
                                ->default(Status::INACTIVE)
                                ->afterStateHydrated(function (Forms\Components\Select $component, $state): void {
                                    if ($state instanceof Status) {
                                        $component->state($state->value);
                                    }
                                })
                                ->dehydrateStateUsing(function ($state): int {
                                    return $state instanceof Status ? $state->value: (int) $state;
                                })
                                ->rule('integer'),
                        ]),
                    Tab::make(__('content::filament.resources.article.form.tabs.content.label'))
                        ->schema([
                            Forms\Components\Textarea::make('lead')
                                ->label(__('content::filament.resources.article.form.fields.lead'))
                                ->required()
                                ->rows(5),
                            RichEditorFormComponents::getRichEditor(
                                __('content::filament.resources.project.form.fields.content'),
                                'content',
                            ),
                            Forms\Components\Repeater::make('key_numbers')
                                ->label(__('content::filament.resources.project.form.fields.key_numbers'))
                                ->default([])
                                ->schema([
                                    Forms\Components\TextInput::make('number')
                                        ->label(__('content::filament.resources.project.form.fields.key_numbers_number'))
                                        ->maxLength(255)
                                        ->live(),
                                    Forms\Components\TextInput::make('header')
                                        ->label(__('content::filament.resources.project.form.fields.key_numbers_header'))
                                        ->maxLength(255),
                                    Forms\Components\TextInput::make('subtitle')
                                        ->label(__('content::filament.resources.project.form.fields.key_numbers_subtitle'))
                                        ->maxLength(255),
                                    Forms\Components\Textarea::make('description')
                                        ->label(__('content::filament.resources.project.form.fields.key_numbers_description'))
                                        ->maxLength(255),
                                ])
                                ->columns(3)
                                ->addActionLabel(__('content::filament.resources.project.form.fields.key_numbers_create_button_label'))
                                ->minItems(0)
                                ->maxItems(20)
                                ->columnSpan('full')
                                ->collapsible()
                                ->collapsed()
                                ->itemLabel(fn (array $state): ?string => $state['number'] ?? null),
                            Forms\Components\Textarea::make('team_description')
                                ->label(__('content::filament.resources.project.form.fields.team_description'))
                                ->rows(3),
                            Forms\Components\Repeater::make('involved_people')
                                ->label(__('content::filament.resources.project.form.fields.involved_people'))
                                ->default([])
                                ->itemLabel(fn (array $state): ?string => $state['name'] ?? null)
                                ->schema([
                                    Forms\Components\TextInput::make('name')
                                        ->label(__('content::filament.resources.project.form.fields.involved_people_name'))
                                        ->maxLength(255),
                                    Forms\Components\TextInput::make('position')
                                        ->label(__('content::filament.resources.project.form.fields.involved_people_position'))
                                        ->maxLength(255),
                                    Forms\Components\FileUpload::make('image')
                                        ->label(__('content::filament.resources.project.form.fields.involved_people_image'))
                                        ->image()
                                        ->directory(static::$contentType . '/involved_people_images')
                                        ->disk('public')
                                        ->downloadable()
                                        ->openable()
                                        ->maxSize(2048),
                                    RichEditorFormComponents::getRichEditor(
                                        __('content::filament.resources.project.form.fields.involved_people_bio'),
                                        'bio',
                                    ),
                                    Forms\Components\TextInput::make('linkedin_url')
                                        ->label(__('content::filament.resources.project.form.fields.involved_people_linkedin_url'))
                                        ->maxLength(255)
                                        ->url(),
                                    Forms\Components\TextInput::make('role')
                                        ->label(__('content::filament.resources.project.form.fields.involved_people_role'))
                                        ->maxLength(255),
                                    Forms\Components\TextInput::make('quote')
                                        ->label(__('content::filament.resources.project.form.fields.involved_people_quote'))
                                        ->maxLength(255),
                                ])
                                ->columns(1)
                                ->addActionLabel(__('content::filament.resources.project.form.fields.involved_people_create_button_label'))
                                ->minItems(0)
                                ->maxItems(20)
                                ->columnSpan('full')
                                ->collapsible()
                                ->collapsed(),
                            Section::make(__('content::filament.resources.project.form.fields.project_metric'))
                                ->schema([
                                    Forms\Components\Select::make('team_composition')
                                        ->label(__('content::filament.resources.project.form.fields.team_composition'))
                                        ->options($dictionaryOptions['position'])
                                        ->multiple()
                                        ->allowHtml()
                                        ->searchable(),
                                    Section::make(__('content::filament.resources.project.form.fields.technology_and_tools'))
                                        ->collapsible()
                                        ->collapsed()
                                        ->schema([
                                            Forms\Components\Select::make('software')
                                                ->label(__('content::filament.resources.project.form.fields.software'))
                                                ->options($dictionaryOptions['software'])
                                                ->allowHtml()
                                                ->multiple()
                                                ->searchable(),
                                            Forms\Components\Select::make('hardware')
                                                ->label(__('content::filament.resources.project.form.fields.hardware'))
                                                ->options($dictionaryOptions['hardware'])
                                                ->allowHtml()
                                                ->multiple()
                                                ->searchable(),
                                            Forms\Components\Select::make('methodologies')
                                                ->label(__('content::filament.resources.project.form.fields.methodologies'))
                                                ->options($dictionaryOptions['methodologies'])
                                                ->allowHtml()
                                                ->multiple()
                                                ->searchable(),
                                        ])->columns(3),
                                ])
                        ]),
                    Tab::make(__('content::filament.resources.article.form.tabs.gallery.label'))
                        ->schema([
                            Forms\Components\Repeater::make('gallery')
                                ->label(__('content::filament.resources.project.form.fields.gallery'))
                                ->schema([
                                    Forms\Components\FileUpload::make('image')
                                        ->label(__('content::filament.resources.project.form.fields.image'))
                                        ->image()
                                        ->directory(static::$contentType . '/gallery')
                                        ->disk('public')
                                        ->downloadable()
                                        ->openable()
                                        ->required()
                                        ->maxSize(4096)
                                        ->imageEditor()
                                        ->imagePreviewHeight('150')
                                        ->columnSpan('full'),
                                    Forms\Components\TextInput::make('image_caption')
                                        ->label(__('content::filament.resources.project.form.fields.image_caption'))
                                        ->maxLength(255)
                                        ->placeholder(__('content::filament.resources.project.form.fields.image_caption_placeholder'))
                                        ->columnSpan('full'),
                                ])
                                ->default([])
                                ->columns(1)
                                ->grid(3)
                                ->addActionLabel(__('content::filament.resources.project.form.fields.gallery_create_button_label'))
                                ->minItems(0)
                                ->maxItems(50)
                                ->columnSpan('full')
                                ->collapsible()
                                ->collapsed(false)
                                ->reorderable()
                                ->itemLabel(fn (array $state): ?string => $state['image_caption'] ?? __('content::filament.resources.project.form.fields.gallery_item_label_default'))
                                ->deleteAction(
                                    fn (Action $action) => $action->requiresConfirmation()
                                ),
                        ]
                    ),
                    SeoFormComponents::getSeoTab(),
                ]),
            ])->columns(1);
    }

    public static function getPages(): array
    {
        return [
            'index'  => \MbeGroup\Content\Filament\Resources\ProjectResource\Pages\ListProjects::route('/'),
            'create' => \MbeGroup\Content\Filament\Resources\ProjectResource\Pages\CreateProject::route('/create'),
            'edit'   => \MbeGroup\Content\Filament\Resources\ProjectResource\Pages\EditProject::route('/{record}/edit'),
        ];
    }
}
