<?php

namespace MbeGroup\Content\Repository;

use MbeGroup\Content\Models\Content;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Collection;
use App\ValueObject\Status;

class ContentRepository
{
    public function paginate(array $filters = [], int $perPage = 15, int $page = 1, string $type): LengthAwarePaginator
    {
        $query = Content::query()
            ->when($type !== '', function ($query) use ($type) {
                $query->where('type', $type);
            })
            ->when($filters['q'] ?? null, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('title', 'like', "%{$search}%");
                });
            })
            ->when($filters['category_ids'] ?? null, function ($query, $categoryIds) {
                $query->whereIn('content_category_id', array_map('intval', explode(',', $categoryIds)));
            })
            ->when($filters['industry_ids'] ?? null, function ($query, $industryIds) {
                $query->whereIn('industry_ids', array_map('intval', explode(',', $industryIds)));
            })
            ->when($filters['specialization_ids'] ?? null, function ($query, $specializationIds) {
                $query->whereIn('specialization_id', array_map('intval', explode(',', $specializationIds)));
            })
            ->when($filters['employer_id'] ?? null, function ($query, $employerId) {
                $query->where('employer_id', $employerId);
            })
            ->when($filters['author_ids'] ?? null, function ($query, $authorIds) {
                $query->whereIn('author_ids', array_map('strval', explode(',', $authorIds)));
            })
            ->when($filters['content_category_ids'] ?? null, function ($query, $contentCategoryIds) {
                $query->whereIn('content_category_id', array_map('intval', explode(',', $contentCategoryIds)));
            })
            ->where('status', Status::ACTIVE)
            ->where(function ($query) {
                $query->whereNull('publication_date')
                    ->orWhere('publication_date', '<=', now());
            })
            ->orderByDesc('created_at');

        return $query->paginate($perPage, ['*'], 'page', $page);
    }

    public function all(): Collection
    {
        return Content::query()
            ->orderByDesc('created_at')
            ->where('status', Status::ACTIVE)
            ->where('publication_date', '<=', now())
            ->get();
    }

    public function findById(string $id): Content
    {
        return Content::where('id', $id)
            ->where('status', Status::ACTIVE)
            ->where('publication_date', '<=', now())
            ->firstOrFail();
    }

    public function findBySlug(string $slug): Content
    {
        return Content::where('slug', $slug)
            ->where('status', Status::ACTIVE)
            ->where('publication_date', '<=', now())
            ->firstOrFail();
    }

    public function findBySlugAndType(string $slug, string $type): Content
    {
        return Content::where('slug', $slug)
            ->where('type', $type)
            ->where('status', Status::ACTIVE)
            ->where('publication_date', '<=', now())
            ->firstOrFail();
    }

    public function create(array $data): Content
    {
        return Content::create($data);
    }

    public function update(Content $content, array $data): Content
    {
        $content->update($data);

        return $content;
    }

    public function delete(Content $content): void
    {
        $content->delete();
    }
}
