<?php

namespace MbeGroup\Content\Controllers\Api\V1;

use App\Helpers\DictionaryHelper;
use App\Http\Controllers\Api\V1\CrudApiController;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use MbeGroup\Content\Resources\ContentResource;
use MbeGroup\Content\Resources\ContentTypeResource;
use MbeGroup\Content\Contracts\ContentServiceInterface;

final class ContentController extends CrudApiController
{
    protected string $resourceName = 'content';

    public function __construct(
        protected ContentServiceInterface $service,
    ) {
    }

    public function types(): JsonResponse
    {
        $filters = request()->only([
            'category',
        ]);

        $types = $this->service->listContentTypes($filters);

        return $this->success(ContentTypeResource::collection($types));
    }

    public function showByType(string $type): JsonResponse
    {
        $slug    = request()->route('slug');
        $content = $this->service->getByType($slug, $type);

        $this->initSeo($content, $this->resourceName . '::seo.' . $type);

        return $this->success(new ContentResource($content));
    }

    public function indexByType(Request $request, string $type): JsonResponse
    {
        $filters = $request->only([
            'q',
            'category_ids',
            'industry_ids',
            'specialization_ids',
            'employer_id',
            'author_ids',
            'content_category_ids',
        ]);
        $perPage = (int) $request->input('limit', self::PER_PAGE_DEFAULT);
        $page    = (int) $request->input('page', 1);

        $contents = $this->service->listByType($filters, $perPage, $page, $type);

        $this->initSeo(null, $this->resourceName . '::seo.' . $type);

        return $this->success(ContentResource::collection($contents),[
            'pagination' => [
                'total'        => $contents->total(),
                'per_page'     => $contents->perPage(),
                'current_page' => $contents->currentPage(),
                'last_page'    => $contents->lastPage(),
            ]
        ]);
    }

    public function index(Request $request): JsonResponse
    {
        $filters = $request->only([
            'q',
            'category_ids',
            'industry_ids',
            'specialization_ids',
            'employer_id',
            'author_ids',
            'content_category_ids',
        ]);

        $perPage  = (int) $request->input('limit', self::PER_PAGE_DEFAULT);
        $page     = (int) $request->input('page', 1);
        $contents = $this->service->listAll($filters, $perPage, $page);
        $seoKey   = $this->service->determineSeoKey($request->input('content_category_ids'));

        $this->initSeo(null, $this->resourceName . '::seo.' . $seoKey);

        return $this->success(ContentResource::collection($contents), [
            'pagination' => [
                'total'        => $contents->total(),
                'per_page'     => $contents->perPage(),
                'current_page' => $contents->currentPage(),
                'last_page'    => $contents->lastPage(),
            ]
        ]);
    }

    public function show(string $type): JsonResponse
    {
        $slug    = request()->route('slug');
        $content = $this->service->get($slug);

        $this->initSeo($content, $this->resourceName . '::seo.' . $type);

        return $this->success(new ContentResource($content));
    }

    public function store(Request $request): JsonResponse
    {
        return $this->notImplemented('store');
    }

    public function update(Request $request, string $id): JsonResponse
    {
        return $this->notImplemented('update');
    }

    public function destroy(string $id): JsonResponse
    {
        return $this->notImplemented('destroy');
    }
}
