<?php

namespace MbeGroup\Employer\Filament\Resources;

use App\Helpers\DictionaryHelper;
use MbeGroup\Employer\Models\Employer;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use MbeGroup\Employer\Filament\Resources\EmployerResource\Pages\ListEmployers;
use MbeGroup\Employer\Filament\Resources\EmployerResource\Pages\CreateEmployer;
use MbeGroup\Employer\Filament\Resources\EmployerResource\Pages\EditEmployer;
use Filament\Actions;
use Filament\Forms;
use App\ValueObject\Status;
use App\Contracts\User\UserLookupServiceInterface;
use Filament\Schemas\Components\Grid;
use MbeGroup\Product\Models\Product;

class EmployerResource extends Resource
{
    protected static ?string $model = Employer::class;

    public static function getNavigationGroup(): string
    {
        return __('employer::filament.navigation.group.label');
    }

    public static function getNavigationLabel(): string
    {
        return __('employer::filament.resources.employer.navigation_label');
    }

    public static function getModelLabel(): string
    {
        return __('employer::filament.resources.employer.model_label');
    }

    public static function getPluralModelLabel(): string
    {
        return __('employer::filament.resources.employer.plural_model_label');
    }

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->schema([
                Tabs::make('Tabs')
                ->tabs([
                    Tab::make(__('employer::filament.resources.employer.form.tabs.basic_info'))
                        ->schema([
                            Forms\Components\TextInput::make('name')
                                ->label(__('employer::filament.resources.employer.form.fields.name'))
                                ->required()
                                ->maxLength(255)
                                ->live(onBlur: true)
                                ->afterStateUpdated(function (string $operation, $state, callable $set) {
                                    if ($operation !== 'create') {
                                        return;
                                    }
                                    $set('slug', \Illuminate\Support\Str::slug($state));
                                }),
                            Forms\Components\TextInput::make('slug')
                                ->label(__('employer::filament.resources.employer.form.fields.slug'))
                                ->required()
                                ->maxLength(255),
                            Forms\Components\Select::make('industry_id')
                                ->label(__('employer::filament.resources.employer.form.fields.industry'))
                                ->options(DictionaryHelper::getNamesByType('industry'))
                                ->required(),
                            Forms\Components\Select::make('company_size_id')
                                ->label(__('employer::filament.resources.employer.form.fields.company_size'))
                                ->options(DictionaryHelper::getNamesByType('company_size'))
                                ->required(),
                            Forms\Components\Select::make('company_type_id')
                                ->label(__('employer::filament.resources.employer.form.fields.company_type'))
                                ->options(DictionaryHelper::getNamesByType('company_type'))
                                ->required(),
                            Grid::make(2)
                                ->schema([
                                    Forms\Components\FileUpload::make('logo')
                                        ->label(__('employer::filament.resources.employer.form.fields.logo'))
                                        ->image()
                                        ->imageEditor()
                                        ->imagePreviewHeight('150')
                                        ->directory('employers/logos')
                                        ->disk('public')
                                        ->downloadable()
                                        ->openable()
                                        ->maxSize(2048),
                                    Forms\Components\FileUpload::make('cover_image')
                                        ->label(__('employer::filament.resources.employer.form.fields.cover_image'))
                                        ->image()
                                        ->imageEditor()
                                        ->imagePreviewHeight('200')
                                        ->directory('employers/covers')
                                        ->disk('public')
                                        ->downloadable()
                                        ->openable()
                                        ->maxSize(5120),
                                ]),
                            Forms\Components\Select::make('account_manager_id')
                                ->label(__('employer::filament.resources.employer.form.fields.account_manager_id'))
                                ->options(function () {
                                    $userLookupService = app()->make(UserLookupServiceInterface::class);
                                    return $userLookupService->getActiveByRoleOptions('admin');
                                })
                                ->searchable()
                                ->preload(),
                            Forms\Components\Select::make('status')
                                    ->label(__('employer::filament.resources.employer.form.fields.status'))
                                    ->options(Status::getLabels())
                                    ->required()
                                    ->default(Status::INACTIVE)
                                    ->afterStateHydrated(function (Forms\Components\Select $component, $state): void {
                                        if ($state instanceof Status) {
                                            $component->state($state->value);
                                        }
                                    })
                                    ->dehydrateStateUsing(function ($state): int {
                                        return $state instanceof Status ? $state->value: (int) $state;
                                    })
                                    ->rule('integer')
                        ]),
                    Tab::make(__('employer::filament.resources.employer.form.tabs.products'))
                        ->schema([
                            Forms\Components\Repeater::make('products')
                                ->label(__('employer::filament.resources.employer.form.fields.products'))
                                ->afterStateHydrated(function (Forms\Components\Repeater $component, $state) {
                                    if (is_array($state)) {
                                        foreach ($state as $index => &$item) {
                                            if (isset($item['product_id']) && (!isset($item['product_name']) || empty($item['product_name']))) {
                                                $product = Product::find($item['product_id']);
                                                if ($product) {
                                                    $item['product_name'] = $product->name;
                                                    $item['unit']         = $product->unit_name;
                                                }
                                            }
                                        }
                                        $component->state($state);
                                    }
                                })
                                ->schema([
                                    Forms\Components\Select::make('product_id')
                                        ->label(__('employer::filament.resources.employer.form.fields.product'))
                                        ->options(function () {
                                            return Product::where('status', Status::ACTIVE)
                                                ->pluck('name', 'id')
                                                ->toArray();
                                        })
                                        ->required()
                                        ->searchable()
                                        ->preload()
                                        ->reactive()
                                        ->afterStateUpdated(function ($state, callable $set, callable $get) {
                                            if ($state) {
                                                $product = Product::find($state);
                                                if ($product) {
                                                    $set('product_name', $product->name);
                                                    $set('unit', $product->unit_name);
                                                }
                                            }
                                        }),
                                    Forms\Components\TextInput::make('quantity')
                                        ->label(__('employer::filament.resources.employer.form.fields.quantity'))
                                        ->numeric()
                                        ->required()
                                        ->default(1)
                                        ->minValue(1)
                                        ->step(1),
                                    Forms\Components\TextInput::make('product_name')
                                        ->label(__('employer::filament.resources.employer.form.fields.product_name'))
                                        ->disabled()
                                        ->dehydrated(true),
                                    Forms\Components\TextInput::make('unit')
                                        ->label(__('employer::filament.resources.employer.form.fields.unit'))
                                        ->disabled()
                                        ->dehydrated(true),
                                    Forms\Components\Select::make('status')
                                        ->label(__('employer::filament.resources.employer.form.fields.status'))
                                        ->options(Status::getLabels())
                                        ->required()
                                        ->default(Status::ACTIVE)
                                        ->afterStateHydrated(function (Forms\Components\Select $component, $state): void {
                                            if ($state instanceof Status) {
                                                $component->state($state->value);
                                            }
                                        })
                                        ->dehydrateStateUsing(function ($state): int {
                                            return $state instanceof Status ? $state->value: (int) $state;
                                        })
                                        ->rule('integer'),
                                ])
                                ->columns(4)
                                ->defaultItems(0)
                                ->addActionLabel(__('employer::filament.resources.employer.form.fields.add_product'))
                                ->reorderable(false)
                                ->collapsible()
                                ->itemLabel(function (array $state): ?string {
                                    if (empty($state['product_name'])) {
                                        return null;
                                    }
                                    $quantity = $state['quantity'] ?? 1;
                                    $unit     = $state['unit'] ?? '';
                                    return "{$state['product_name']} - x{$quantity}";
                                })
                        ]),
                ])
            ])->columns(1);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label('ID')
                    ->sortable(),
                Tables\Columns\TextColumn::make('name')
                    ->label(__('employer::filament.resources.employer.table.columns.name'))
                    ->searchable(),
                Tables\Columns\TextColumn::make('slug')
                    ->label(__('employer::filament.resources.employer.table.columns.slug'))
                    ->searchable(),
                Tables\Columns\IconColumn::make('status')
                    ->label(__('employer::filament.resources.employer.table.columns.status'))
                    ->state(fn ($record) => Status::int($record->status))
                    ->icon(fn ($state) => Status::icon($state))
                    ->color(fn ($state) => Status::color($state))
                    ->tooltip(fn ($state) => Status::label($state))
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                ->label(__('employer::filament.resources.employer.table.columns.created_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                ->label(__('employer::filament.resources.employer.table.columns.updated_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                  //
            ])
            ->recordActions([
                Actions\EditAction::make(),
                Actions\DeleteAction::make(),
            ])
            ->toolbarActions([
                Actions\BulkActionGroup::make([
                    Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->description('Version: ' . \Composer\InstalledVersions::getVersion('mbegroup/employer'));
    }

    public static function getRelations(): array
    {
        return [

        ];
    }

    public static function getPages(): array
    {
        return [
            'index'  => ListEmployers::route('/'),
            'create' => CreateEmployer::route('/create'),
            'edit'   => EditEmployer::route('/{record}/edit'),
        ];
    }
}
