<?php

namespace MbeGroup\Employer\Filament\Resources\EmployerResource\Pages;

use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use MbeGroup\Employer\Filament\Resources\EmployerResource;
use MbeGroup\Product\Models\ProductStatistics;
use MbeGroup\Product\Models\Product;
use App\ValueObject\Status;

class EditEmployer extends EditRecord
{
    protected static string $resource = EmployerResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\DeleteAction::make(),
        ];
    }

    protected function getFormActions(): array
    {
        return [
            $this->getSaveFormAction(),
            Actions\Action::make('saveAndBack')
                ->label(__('employer::filament.resources.employer.form.description.saveAndBack'))
                ->color('gray')
                ->action(function () {
                    $this->save();
                    $this->redirect($this->getResource()::getUrl('index'));
                })
                ->keyBindings(['mod+shift+s']),
            $this->getCancelFormAction(),
        ];
    }

    protected function afterSave(): void
    {
        $this->syncProductStatistics();
    }

    protected function mutateFormDataBeforeFill(array $data): array
    {
        $record = $this->getRecord();

        $activeStatistics = ProductStatistics::where('employer_id', $record->getKey())
            ->where('status', Status::ACTIVE)
            ->with('product')
            ->get();

        $products = [];
        foreach ($activeStatistics as $statistic) {
            $products[] = [
                'product_id'   => $statistic->product_id,
                'quantity'     => $statistic->quantity,
                'product_name' => $statistic->product->name ?? '',
                'unit'         => $statistic->product->unit_name ?? '',
                'status'       => $statistic->status,
            ];
        }

        $data['products'] = $products;

        return $data;
    }

    private function syncProductStatistics(): void
    {
        $record          = $this->getRecord();
        $currentProducts = collect($this->data['products'] ?? []);

        $existingStatistics = ProductStatistics::where('employer_id', $record->getKey())
            ->where('status', Status::ACTIVE)
            ->get();

        $currentProductIds = $currentProducts->pluck('product_id')->filter()->toArray();

        $existingStatistics->each(function ($statistic) use ($currentProductIds) {
            if (!in_array($statistic->product_id, $currentProductIds)) {
                $statistic->update([
                    'status'     => Status::INACTIVE,
                    'updated_at' => now()
                ]);
            }
        });

        foreach ($currentProducts as $productData) {
            if (isset($productData['product_id'])) {
                $product = Product::find($productData['product_id']);
                if ($product) {
                    $existingStatistic = $existingStatistics->firstWhere('product_id', $productData['product_id']);

                    if ($existingStatistic) {
                        $existingStatistic->update([
                            'quantity'   => $productData['quantity'] ?? 1,
                            'price'      => $product->price_net * ($productData['quantity'] ?? 1),
                            'status'     => $productData['status'] ?? Status::ACTIVE,
                            'updated_at' => now()
                        ]);
                    } else {
                        $inactiveStatistic = ProductStatistics::where('employer_id', $record->getKey())
                            ->where('product_id', $productData['product_id'])
                            ->where('status', Status::INACTIVE)
                            ->first();

                        if ($inactiveStatistic) {
                            $inactiveStatistic->update([
                                'quantity'   => $productData['quantity'] ?? 1,
                                'price'      => $product->price_net * ($productData['quantity'] ?? 1),
                                'status'     => $productData['status'] ?? Status::ACTIVE,
                                'updated_at' => now()
                            ]);
                        } else {
                            ProductStatistics::create([
                                'employer_id' => $record->getKey(),
                                'product_id'  => $productData['product_id'],
                                'quantity'    => $productData['quantity'] ?? 1,
                                'price'       => $product->price_net * ($productData['quantity'] ?? 1),
                                'valid_from'  => now(),
                                'valid_to'    => now()->addMonth(),
                                'status'      => $productData['status'] ?? Status::ACTIVE,
                            ]);
                        }
                    }
                }
            }
        }
    }
}
