<?php

namespace Tests\Unit;

use MbeGroup\Employer\Models\Employer;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

class EmployerTest extends TestCase
{
    /**
     * Test tworzenia nowej instancji Employer
     */
    public function test_can_create_employer_instance(): void
    {
        $employer = new Employer();
        
        $this->assertInstanceOf(Employer::class, $employer);
    }

    /**
     * Test sprawdzający czy klasa ma odpowiednie właściwości
     */
    public function test_employer_class_properties(): void
    {
        $reflection = new ReflectionClass(Employer::class);
        
        // Sprawdzenie czy klasa dziedziczy po MongoDB\Laravel\Eloquent\Model
        $this->assertTrue($reflection->hasProperty('connection'));
        $this->assertTrue($reflection->hasProperty('collection'));
        $this->assertTrue($reflection->hasProperty('fillable'));
        $this->assertTrue($reflection->hasProperty('casts'));
    }

    /**
     * Test sprawdzający wypełnialne pola
     */
    public function test_fillable_attributes(): void
    {
        $employer = new Employer();
        $reflection = new ReflectionClass($employer);
        
        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($employer);
        
        $expectedFillable = [
            'name',
            'slug',
            'industry_id',
            'company_size_id',
            'company_type_id',
            'logo',
            'cover_image',
            'account_manager_id',
            'status',
            'locations',
            'product_ids',
            'products',
            'email',
            'phone',
            'website',
            'fv_name',
            'fv_nip',
            'fv_address',
            'job_offer_template',
            'publication_settings',
        ];
        $this->assertEquals($expectedFillable, $fillable);
    }

    /**
     * Test sprawdzający connection
     */
    public function test_mongodb_connection(): void
    {
        $employer = new Employer();
        $reflection = new ReflectionClass($employer);
        
        $connectionProperty = $reflection->getProperty('connection');
        $connectionProperty->setAccessible(true);
        $connection = $connectionProperty->getValue($employer);
        
        $this->assertEquals('mongodb', $connection);
    }

    /**
     * Test sprawdzający nazwę kolekcji
     */
    public function test_collection_name(): void
    {
        $employer = new Employer();
        $reflection = new ReflectionClass($employer);
        
        $collectionProperty = $reflection->getProperty('collection');
        $collectionProperty->setAccessible(true);
        $collection = $collectionProperty->getValue($employer);
        
        $this->assertEquals('employers', $collection);
    }

    /**
     * Test sprawdzający casts
     */
    public function test_casts_configuration(): void
    {
        $employer = new Employer();
        $reflection = new ReflectionClass($employer);
        
        $castsProperty = $reflection->getProperty('casts');
        $castsProperty->setAccessible(true);
        $casts = $castsProperty->getValue($employer);
        
        $expectedCasts = [
            'status' => \App\ValueObject\Status::class,
            'locations' => 'array',
        ];
        
        $this->assertEquals($expectedCasts, $casts);
    }

    /**
     * Test sprawdzający namespace klasy
     */
    public function test_class_namespace(): void
    {
        $employer = new Employer();
        $reflection = new ReflectionClass($employer);
        
        $this->assertEquals('MbeGroup\Employer\Models', $reflection->getNamespaceName());
        $this->assertEquals('Employer', $reflection->getShortName());
    }

    /**
     * Test czy klasa implementuje odpowiednie metody (sprawdzenie czy dziedziczy po Model)
     */
    public function test_extends_mongodb_model(): void
    {
        $employer = new Employer();
        
        // Sprawdzenie czy klasa dziedziczy po MongoDB Laravel Eloquent Model
        $this->assertInstanceOf(\MongoDB\Laravel\Eloquent\Model::class, $employer);
    }

    /**
     * Test konstruktora z danymi
     */
    public function test_constructor_with_attributes(): void
    {
        $attributes = [
            'name' => 'Test Employer',
            'slug' => 'test-employer',
            'industry_id' => 'industry_123',
            'company_size_id' => 'size_456',
            'company_type_id' => 'type_789',
        ];

        // Test czy konstruktor przyjmuje tablicę atrybutów
        $employer = new Employer($attributes);
        $this->assertInstanceOf(Employer::class, $employer);
    }

    /**
     * Test sprawdzający czy wszystkie wymagane właściwości są zdefiniowane
     */
    public function test_all_required_properties_are_defined(): void
    {
        $employer = new Employer();
        $reflection = new ReflectionClass($employer);
        
        $expectedProperties = ['connection', 'collection', 'fillable', 'casts'];
        
        foreach ($expectedProperties as $property) {
            $this->assertTrue(
                $reflection->hasProperty($property),
                "Właściwość {$property} nie została znaleziona w klasie Employer"
            );
        }
    }

    /**
     * Test sprawdzający dostępność metod relacyjnych
     */
    public function test_relationship_methods_exist(): void
    {
        $employer = new Employer();
        $reflection = new ReflectionClass($employer);
        
        $expectedMethods = ['industry', 'companySize', 'companyType', 'productOrders', 'products'];
        
        foreach ($expectedMethods as $method) {
            $this->assertTrue(
                $reflection->hasMethod($method),
                "Metoda {$method} nie została znaleziona w klasie Employer"
            );
        }
    }

    /**
     * Test sprawdzający czy metody relacji belongsTo zwracają odpowiedni typ przez refleksję
     */
    public function test_belongs_to_relationships(): void
    {
        $reflection = new ReflectionClass(Employer::class);
        
        // Test czy metody relacji istnieją i są publiczne
        $industryMethod = $reflection->getMethod('industry');
        $this->assertTrue($industryMethod->isPublic());
        
        $companySizeMethod = $reflection->getMethod('companySize');
        $this->assertTrue($companySizeMethod->isPublic());
        
        $companyTypeMethod = $reflection->getMethod('companyType');
        $this->assertTrue($companyTypeMethod->isPublic());
        
        // Sprawdzenie czy metody nie wymagają parametrów
        $this->assertCount(0, $industryMethod->getParameters());
        $this->assertCount(0, $companySizeMethod->getParameters());
        $this->assertCount(0, $companyTypeMethod->getParameters());
    }

    /**
     * Test sprawdzający czy metody relacji hasMany istnieją
     */
    public function test_has_many_relationships(): void
    {
        $reflection = new ReflectionClass(Employer::class);
        
        // Test czy metoda productOrders istnieje i jest publiczna
        $productOrdersMethod = $reflection->getMethod('productOrders');
        $this->assertTrue($productOrdersMethod->isPublic());
        
        // Sprawdzenie czy metoda nie wymaga parametrów
        $this->assertCount(0, $productOrdersMethod->getParameters());
    }

    /**
     * Test sprawdzający czy metody relacji hasManyThrough istnieją
     */
    public function test_has_many_through_relationships(): void
    {
        $reflection = new ReflectionClass(Employer::class);
        
        // Test czy metoda products istnieje i jest publiczna
        $productsMethod = $reflection->getMethod('products');
        $this->assertTrue($productsMethod->isPublic());
        
        // Sprawdzenie czy metoda nie wymaga parametrów
        $this->assertCount(0, $productsMethod->getParameters());
    }

    /**
     * Test sprawdzający czy wszystkie pola fillable są stringami lub właściwymi typami
     */
    public function test_fillable_field_types(): void
    {
        $employer = new Employer();
        $reflection = new ReflectionClass($employer);
        
        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($employer);
        
        // Sprawdzenie czy wszystkie fillable są stringami
        foreach ($fillable as $field) {
            $this->assertIsString($field, "Pole fillable '{$field}' powinno być stringiem");
        }
        
        // Sprawdzenie czy nie ma zduplikowanych pól
        $this->assertEquals(count($fillable), count(array_unique($fillable)), 'Nie powinno być zduplikowanych pól w fillable');
    }

    /**
     * Test sprawdzający czy istnieją wymagane pola biznesowe
     */
    public function test_required_business_fields_are_fillable(): void
    {
        $employer = new Employer();
        $reflection = new ReflectionClass($employer);
        
        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($employer);
        
        $requiredBusinessFields = ['name', 'slug', 'status'];
        
        foreach ($requiredBusinessFields as $field) {
            $this->assertContains($field, $fillable, "Wymagane pole biznesowe '{$field}' nie jest w fillable");
        }
    }
}