<?php

namespace MbeGroup\Employer\Filament\Resources;

use App\Helpers\DictionaryHelper;
use MbeGroup\Employer\Models\Employer;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use MbeGroup\Employer\Filament\Resources\EmployerResource\Pages\ListEmployers;
use MbeGroup\Employer\Filament\Resources\EmployerResource\Pages\CreateEmployer;
use MbeGroup\Employer\Filament\Resources\EmployerResource\Pages\EditEmployer;
use Filament\Actions;
use Filament\Forms;
use App\ValueObject\Status;
use App\Contracts\User\UserLookupServiceInterface;
use Filament\Forms\FormsComponent;
use Filament\Schemas\Components\Grid;
use MbeGroup\Product\Models\Product;
use Tapp\FilamentGoogleAutocomplete\Forms\Components\GoogleAutocomplete;
use App\Filament\Components\OpenStreetMapFormComponents;

class EmployerResource extends Resource
{
    protected static ?string $model = Employer::class;

        /**
     * Configure Status field with common hydration and dehydration logic
     */
    private static function configureStatusField(Forms\Components\Select $field): Forms\Components\Select
    {
        return $field
            ->afterStateHydrated(function (Forms\Components\Select $component, $state): void {
                if ($state instanceof Status) {
                    $component->state($state->value);
                }
            })
            ->dehydrateStateUsing(function ($state): int {
                return $state instanceof Status ? $state->value: (int) $state;
            })
            ->rule('integer');
    }

    public static function getNavigationGroup(): string
    {
        return __('employer::filament.navigation.group.label');
    }

    public static function getNavigationLabel(): string
    {
        return __('employer::filament.resources.employer.navigation_label');
    }

    public static function getModelLabel(): string
    {
        return __('employer::filament.resources.employer.model_label');
    }

    public static function getPluralModelLabel(): string
    {
        return __('employer::filament.resources.employer.plural_model_label');
    }

    public static function form(Schema $schema): Schema
    {
        $statusOptions     = Status::getLabels();
        $dictionaryOptions = [
            'industry'                    => DictionaryHelper::getHierarchicalNamesByType('industry'),
            'company_size'                => DictionaryHelper::getHierarchicalNamesByType('company_size'),
            'company_type'                => DictionaryHelper::getHierarchicalNamesByType('company_type'),
            'job_offer_variable_type'     => DictionaryHelper::getHierarchicalNamesByType('job_offer_variable_type'),
            'job_offer_refresh_frequency' => DictionaryHelper::getHierarchicalNamesByType('job_offer_refresh_frequency'),
            'application_method'          => DictionaryHelper::getSlugsForOptionsByType('application_method'),
        ];

        $activeProducts = Product::where('status', Status::ACTIVE)
            ->pluck('name', 'id')
            ->toArray();

        $userLookupService = app()->make(UserLookupServiceInterface::class);
        $adminUsers        = $userLookupService->getActiveByRoleOptions('admin');
        $operation         = null;

        return $schema
            ->schema([
                Tabs::make('Tabs')
                ->persistTabInQueryString()
                ->tabs([
                    Tab::make(__('employer::filament.resources.employer.form.tabs.basic_info'))
                        ->schema([
                            Forms\Components\TextInput::make('name')
                                ->label(__('employer::filament.resources.employer.form.fields.name'))
                                ->required()
                                ->maxLength(255)
                                ->live(onBlur: true)
                                ->afterStateUpdated(function (string $operation, $state, callable $set) {
                                    if ($operation !== 'create') {
                                        return;
                                    }
                                    $set('slug', \Illuminate\Support\Str::slug($state));
                                }),
                            Forms\Components\TextInput::make('slug')
                                ->label(__('employer::filament.resources.employer.form.fields.slug'))
                                ->required()
                                ->maxLength(255),
                            Forms\Components\Select::make('industry_id')
                                ->label(__('employer::filament.resources.employer.form.fields.industry'))
                                ->options($dictionaryOptions['industry'])
                                ->allowHtml()
                                ->searchable()
                                ->required(),
                            Forms\Components\Select::make('company_size_id')
                                ->label(__('employer::filament.resources.employer.form.fields.company_size'))
                                ->options($dictionaryOptions['company_size'])
                                ->allowHtml()
                                ->searchable()
                                ->required(),
                            Forms\Components\Select::make('company_type_id')
                                ->label(__('employer::filament.resources.employer.form.fields.company_type'))
                                ->options($dictionaryOptions['company_type'])
                                ->allowHtml()
                                ->searchable()
                                ->required(),
                            Grid::make(2)
                                ->schema([
                                    Forms\Components\FileUpload::make('logo')
                                        ->label(__('employer::filament.resources.employer.form.fields.logo'))
                                        ->image()
                                        ->imageEditor()
                                        ->imagePreviewHeight('150')
                                        ->directory('employers/logos')
                                        ->disk('public')
                                        ->downloadable()
                                        ->openable()
                                        ->maxSize(2048),
                                    Forms\Components\FileUpload::make('cover_image')
                                        ->label(__('employer::filament.resources.employer.form.fields.cover_image'))
                                        ->image()
                                        ->imageEditor()
                                        ->imagePreviewHeight('200')
                                        ->directory('employers/covers')
                                        ->disk('public')
                                        ->downloadable()
                                        ->openable()
                                        ->maxSize(5120),
                                    Forms\Components\FileUpload::make('banner_image')
                                        ->label(__('employer::filament.resources.employer.form.fields.banner_image'))
                                        ->image()
                                        ->imageEditor()
                                        ->imagePreviewHeight('200')
                                        ->directory('employers/banners')
                                        ->disk('public')
                                        ->downloadable()
                                        ->openable()
                                        ->maxSize(5120),
                                ]),
                            Forms\Components\Select::make('account_manager_id')
                                ->label(__('employer::filament.resources.employer.form.fields.account_manager_id'))
                                ->options($adminUsers)
                                ->searchable()
                                ->preload(),
                            self::configureStatusField(
                                Forms\Components\Select::make('status')
                                    ->label(__('employer::filament.resources.employer.form.fields.status'))
                                    ->options($statusOptions)
                                    ->required()
                                    ->default(Status::INACTIVE)
                            )
                        ]),
                    Tab::make(__('employer::filament.resources.employer.form.tabs.location_and_contact'))
                        ->schema([
                            OpenStreetMapFormComponents::getOpenStreetMapFields(),
                        ]),
                    Tab::make(__('employer::filament.resources.employer.form.tabs.fv_info'))
                        ->schema([
                            Forms\Components\TextInput::make('fv_name')
                                ->label(__('employer::filament.resources.employer.form.fields.fv_name'))
                                ->maxLength(255),
                            Forms\Components\TextInput::make('fv_nip')
                                ->label(__('employer::filament.resources.employer.form.fields.fv_nip'))
                                ->maxLength(20),
                            Forms\Components\TextInput::make('fv_address')
                                ->label(__('employer::filament.resources.employer.form.fields.fv_address'))
                                ->maxLength(500),
                        ]),
                    Tab::make(__('employer::filament.resources.employer.form.tabs.job_offer_template'))
                        ->schema([
                            Forms\Components\Repeater::make('job_offer_template')
                                ->label(__('employer::filament.resources.employer.form.fields.job_offer_template'))
                                ->lazy()
                                ->schema([
                                    Forms\Components\TextInput::make('job_offer_template_name')
                                         ->label(__('employer::filament.resources.employer.form.fields.job_offer_template_name'))
                                         ->maxLength(255)
                                         ->required(),
                                    Grid::make(2)
                                        ->schema([
                                            Forms\Components\CodeEditor::make('job_offer_template_html')
                                                ->label(__('employer::filament.resources.employer.form.fields.job_offer_template_html')),
                                            Forms\Components\CodeEditor::make('job_offer_template_css')
                                                ->label(__('employer::filament.resources.employer.form.fields.job_offer_template_css'))
                                        ]),
                                    Forms\Components\Checkbox::make('job_offer_is_default_template')
                                        ->label(__('employer::filament.resources.employer.form.fields.job_offer_is_default_template'))
                                        ->live()
                                        ->afterStateUpdated(function ($state, callable $get, callable $set) {
                                            if ($state) {
                                                $allTemplates = $get('../../job_offer_template') ?? [];
                                                foreach ($allTemplates as $index => $template) {
                                                    $set("../../job_offer_template.{$index}.job_offer_is_default_template", false);
                                                }
                                                $set('job_offer_is_default_template', true);
                                            }
                                        }),
                                    Forms\Components\Repeater::make('job_offer_template_variables')
                                        ->label(__('employer::filament.resources.employer.form.fields.job_offer_template_variables'))
                                        ->schema([
                                            Forms\Components\TextInput::make('variable_name')
                                                ->label(__('employer::filament.resources.employer.form.fields.variable_name'))
                                                ->maxLength(255),
                                            Forms\Components\Select::make('variable_type')
                                                ->label(__('employer::filament.resources.employer.form.fields.variable_type'))
                                                ->options(DictionaryHelper::getNamesByType('job_offer_variable_type'))
                                        ])
                                        ->columns(2)
                                        ->defaultItems(0)
                                        ->addActionLabel(__('employer::filament.resources.employer.form.fields.add_variable'))
                                ])
                                ->reorderable(false)
                                ->collapsible()
                                ->collapsed(true)
                                ->cloneable(false)
                                ->itemLabel(function (array $state): ?string {
                                    if (empty($state['job_offer_template_name'])) {
                                        return null;
                                    }
                                    return $state['job_offer_template_name'] . ($state['job_offer_is_default_template'] ?? false ? ' (Default)' : '');
                                })
                        ]),
                    Tab::make(__('Obrazy szablonu'))
                        ->visible(fn (callable $get, string $operation) => $operation === 'edit')
                        ->schema([
                            Forms\Components\Repeater::make('gallery_images')
                                ->label(__('employer::filament.resources.employer.form.fields.gallery_images'))
                                ->lazy()
                                ->schema([
                                    Grid::make(1)
                                        ->schema([
                                            Forms\Components\FileUpload::make('image')
                                                ->label(__('employer::filament.resources.employer.form.fields.image'))
                                                ->image()
                                                ->imageEditor()
                                                ->imagePreviewHeight('200')
                                                ->directory(function ($get, ?Employer $record) {
                                                    $employerId = $record?->id ?? $get('id');
                                                    return $employerId ? 'employers/' . $employerId . '/templates' : 'employers/templates';
                                                })
                                                ->disk('public')
                                                ->downloadable()
                                                ->openable()
                                                ->maxSize(5120)
                                                ->columnSpanFull(),

                                            Forms\Components\TextInput::make('caption')
                                                ->label(__('employer::filament.resources.employer.form.fields.caption'))
                                                ->placeholder('Dodaj opis obrazu...')
                                                ->maxLength(255)
                                                ->columnSpanFull(),
                                        ])
                                ])
                                ->grid(3)
                                ->defaultItems(0)
                                ->addActionLabel(__('employer::filament.resources.employer.form.fields.add_gallery_image'))
                                ->reorderable(true)
                                ->collapsible()
                                ->collapsed(false)
                                ->cloneable(true)
                                ->itemLabel(function (array $state): ?string {
                                    return $state['caption'] ?? 'Obraz bez opisu';
                                })
                        ]),
                    Tab::make(__('employer::filament.resources.employer.form.tabs.products'))
                        ->schema([
                            Forms\Components\Repeater::make('products')
                                ->label(__('employer::filament.resources.employer.form.fields.products'))
                                ->lazy()
                                ->afterStateHydrated(function (Forms\Components\Repeater $component, $state) {
                                    if (is_array($state)) {
                                        foreach ($state as $index => &$item) {
                                            if (isset($item['product_id']) && (!isset($item['product_name']) || empty($item['product_name']))) {
                                                $product = Product::find($item['product_id']);
                                                if ($product) {
                                                    $item['product_name']          = $product->name;
                                                    $item['validity_period_months'] = $product->validity_period_months;
                                                    $item['product_life_days'] = $product->product_life_days;
                                                }
                                            }
                                        }
                                        $component->state($state);
                                    }
                                })
                                ->schema([
                                    Forms\Components\Select::make('product_id')
                                        ->label(__('employer::filament.resources.employer.form.fields.product'))
                                        ->options($activeProducts)
                                        ->required()
                                        ->searchable()
                                        ->preload()
                                        ->reactive()
                                        ->afterStateUpdated(function ($state, callable $set, callable $get) {
                                            if ($state) {
                                                $product = Product::find($state);
                                                if ($product) {
                                                    $set('product_name', $product->name);
                                                    $set('validity_period_months', $product->validity_period_months);
                                                    $set('product_life_days', $product->product_life_days);
                                                    $set('validity_period_start',\Carbon\Carbon::now()->format('Y-m-d'));
                                                    $set('validity_period_end', \Carbon\Carbon::now()->addMonths($product->validity_period_months)->format('Y-m-d'));
                                                }
                                            }
                                        }),
                                    Forms\Components\Hidden::make('validity_period_months')
                                        ->dehydrated(true),
                                    Forms\Components\Hidden::make('product_life_days')
                                        ->dehydrated(true),
                                    Forms\Components\Hidden::make('item_id')
                                        ->default(fn () => (string) \Illuminate\Support\Str::uuid())
                                        ->dehydrated(true),
                                    Forms\Components\TextInput::make('quantity')
                                        ->label(__('employer::filament.resources.employer.form.fields.quantity'))
                                        ->numeric()
                                        ->required()
                                        ->default(1)
                                        ->minValue(1)
                                        ->step(1),
                                    Forms\Components\TextInput::make('product_name')
                                        ->label(__('employer::filament.resources.employer.form.fields.product_name'))
                                        ->readOnly()
                                        ->dehydrated(true),
                                    Forms\Components\DatePicker::make('validity_period_start')
                                        ->label(__('employer::filament.resources.employer.form.fields.validity_period_start'))
                                        ->dehydrated(true)
                                        ->reactive()
                                        ->readOnly(),
                                    Forms\Components\DatePicker::make('validity_period_end')
                                        ->label(__('employer::filament.resources.employer.form.fields.validity_period_end'))
                                        ->readOnly()
                                        ->dehydrated(true)
                                        ->reactive(),
                                    Forms\Components\Select::make('status')
                                        ->label(__('employer::filament.resources.employer.form.fields.status'))
                                        ->options(Status::getLabels())
                                        ->required()
                                        ->default(Status::ACTIVE)
                                        ->afterStateHydrated(function (Forms\Components\Select $component, $state): void {
                                            if ($state instanceof Status) {
                                                $component->state($state->value);
                                            }
                                        })
                                        ->dehydrateStateUsing(function ($state): int {
                                            return $state instanceof Status ? $state->value: (int) $state;
                                        })
                                        ->rule('integer'),
                                ])
                                ->columns(4)
                                ->defaultItems(0)
                                ->addActionLabel(__('employer::filament.resources.employer.form.fields.add_product'))
                                ->reorderable(false)
                                ->collapsible()
                                ->collapsed()
                                ->itemLabel(function (array $state): ?string {
                                    if (empty($state['product_name'])) {
                                        return null;
                                    }
                                    $quantity = $state['quantity'] ?? 1;
                                    $unit     = $state['unit'] ?? '';
                                    return "{$state['product_name']} - x{$quantity}";
                                })
                        ]),
                        Tab::make(__('employer::filament.resources.employer.form.tabs.publication_settings'))
                            ->schema([
                                Tabs::make('PublicationTabs')
                                    ->tabs([
                                        Tab::make(__('employer::filament.resources.employer.form.tabs.apply_options'))
                                            ->schema([
                                                Forms\Components\Select::make('publication_settings.apply_options.application_method')
                                                    ->label(__('employer::filament.resources.employer.form.fields.application_method'))
                                                    ->options($dictionaryOptions['application_method'])
                                                    ->allowHtml()
                                                    ->searchable()
                                                    ->live()
                                                    ->afterStateUpdated(function ($state, callable $set) {
                                                        if ($state !== 'email') {
                                                            $set('publication_settings.apply_options.application_email', null);
                                                        }
                                                        if ($state !== 'phone') {
                                                            $set('publication_settings.apply_options.application_phone', null);
                                                        }
                                                        if ($state !== 'url') {
                                                            $set('publication_settings.apply_options.application_url', null);
                                                        }
                                                    }),
                                                Forms\Components\TextInput::make('publication_settings.apply_options.application_link_reference')
                                                    ->label(__('employer::filament.resources.employer.form.fields.application_link_reference'))
                                                    ->visible(fn (callable $get) => $get('publication_settings.apply_options.application_method') === 'employer-link'),
                                                Forms\Components\TextInput::make('publication_settings.apply_options.application_link_reference_value')
                                                    ->label(__('employer::filament.resources.employer.form.fields.application_link_reference_value'))
                                                    ->visible(fn (callable $get) => $get('publication_settings.apply_options.application_method') === 'employer-link'),
                                                Forms\Components\TextInput::make('publication_settings.apply_options.application_email')
                                                    ->label(__('employer::filament.resources.employer.form.fields.application_email'))
                                                    ->email()
                                                    ->visible(fn (callable $get) => $get('publication_settings.apply_options.application_method') === 'kwf-form')
                                                    ->maxLength(255),
                                                Forms\Components\Checkbox::make('publication_settings.apply_options.application_remote_recruitment')
                                                    ->label(__('employer::filament.resources.employer.form.fields.application_remote_recruitment'))
                                                    ->visible(fn (callable $get) => $get('publication_settings.apply_options.application_method') === 'kwf-form')
                                                    ->default(false),
                                            ]),
                                        Tab::make(__('employer::filament.resources.employer.form.tabs.job_offers_refresh'))
                                            ->schema([
                                                Forms\Components\Select::make('publication_settings.job_offers_refresh.refresh_frequency')
                                                    ->label(__('employer::filament.resources.employer.form.fields.job_offer_refresh_frequency'))
                                                    ->options($dictionaryOptions['job_offer_refresh_frequency'])
                                                    ->allowHtml()
                                                    ->searchable(),
                                                Forms\Components\Checkbox::make('publication_settings.job_offers_refresh.frequency_update')
                                                    ->label(__('employer::filament.resources.employer.form.fields.job_offer_refresh_frequency_update'))
                                                    ->default(false),
                                            ]),
                                        Tab::make(__('employer::filament.resources.employer.form.tabs.seo'))
                                            ->schema([
                                                Forms\Components\TextInput::make('publication_settings.seo.seo_title')
                                                    ->label(__('employer::filament.resources.employer.form.fields.seo_title'))
                                                    ->maxLength(255),
                                                Forms\Components\Textarea::make('publication_settings.seo.seo_description')
                                                    ->label(__('employer::filament.resources.employer.form.fields.seo_description'))
                                                    ->maxLength(500)
                                                    ->rows(3),
                                                Forms\Components\TagsInput::make('publication_settings.seo.seo_keywords')
                                                    ->label(__('employer::filament.resources.employer.form.fields.seo_keywords')),
                                                Forms\Components\Checkbox::make('publication_settings.seo.seo_noindex')
                                                    ->label(__('employer::filament.resources.employer.form.fields.seo_noindex'))
                                            ]),
                                    ])
                                    ->columnSpanFull(),
                            ]),
                ])
            ])->columns(1);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label('ID')
                    ->sortable(),
                Tables\Columns\TextColumn::make('name')
                    ->label(__('employer::filament.resources.employer.table.columns.name'))
                    ->searchable(),
                Tables\Columns\TextColumn::make('slug')
                    ->label(__('employer::filament.resources.employer.table.columns.slug'))
                    ->searchable(),
                Tables\Columns\IconColumn::make('status')
                    ->label(__('employer::filament.resources.employer.table.columns.status'))
                    ->state(fn ($record) => Status::int($record->status))
                    ->icon(fn ($state) => Status::icon($state))
                    ->color(fn ($state) => Status::color($state))
                    ->tooltip(fn ($state) => Status::label($state))
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                ->label(__('employer::filament.resources.employer.table.columns.created_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                ->label(__('employer::filament.resources.employer.table.columns.updated_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                              //
            ])
            ->recordActions([
                Actions\EditAction::make(),
                Actions\DeleteAction::make(),
            ])
            ->toolbarActions([
                Actions\BulkActionGroup::make([
                    Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->description('Version: ' . \Composer\InstalledVersions::getVersion('mbegroup/employer'));
    }

    public static function getRelations(): array
    {
        return [

        ];
    }

    public static function getPages(): array
    {
        return [
            'index'  => ListEmployers::route('/'),
            'create' => CreateEmployer::route('/create'),
            'edit'   => EditEmployer::route('/{record}/edit'),
        ];
    }
}
