<?php
namespace MbeGroup\Employer\Models;

use MongoDB\Laravel\Eloquent\Model;
use App\ValueObject\Status;
use App\Models\Dictionary;
use Illuminate\Support\Facades\Cache;

class Employer extends Model
{
    protected $connection = 'mongodb';
    protected $collection = 'employers';
    protected $fillable   = [
        'name',
        'slug',
        'industry_id',
        'company_size_id',
        'company_type_id',
        'logo',
        'banner_image',
        'cover_image',
        'account_manager_id',
        'status',
        'locations',
        'product_ids',
        'products',
        'email',
        'phone',
        'website',
        'fv_name',
        'fv_nip',
        'fv_address',
        'job_offer_template',
        'publication_settings',
        'gallery_images'
    ];

    protected $casts = [
        'status'         => Status::class,
        'locations'      => 'array',
        'gallery_images' => 'array',
    ];

    public function getLogoUrlAttribute()
    {
        return $this->logo ? url('storage/' . ltrim($this->logo, '/')): null;
    }

    public function getBannerImageUrlAttribute()
    {
        return $this->banner_image ? url('storage/' . ltrim($this->banner_image, '/')): null;
    }
    public function getCoverImageUrlAttribute()
    {
        return $this->cover_image ? url('storage/' . ltrim($this->cover_image, '/')): null;
    }

    public function industry()
    {
        return $this->belongsTo(Dictionary::class, 'industry_id');
    }

    public function companySize()
    {
        return $this->belongsTo(Dictionary::class, 'company_size_id');
    }

    public function companyType()
    {
        return $this->belongsTo(Dictionary::class, 'company_type_id');
    }

      /**
     * Relacja z zamówieniami produktów (tabela łącząca)
     */
    public function productOrders()
    {
        return $this->hasMany(\MbeGroup\Product\Models\ProductStatistics::class);
    }

      /**
     * Pobiera cache'owane opcje pracodawców dla selectów
     */
    public static function getSelectOptions(): array
    {
        return Cache::remember('employer_select_options', 3600, function () {
            return static::where('status', Status::ACTIVE)
                ->get()
                ->mapWithKeys(fn ($employer) => [(string) $employer->_id => $employer->name])
                ->toArray();
        });
    }

      /**
     * Wyszukuje pracodawców dla select z cache'em
     */
    public static function searchForSelect(string $search): array
    {
        $cacheKey = 'employer_search_' . md5($search);
        return Cache::remember($cacheKey, 300, function () use ($search) {
            return static::where('name', 'like', "%{$search}%")
                ->where('status', Status::ACTIVE)
                ->limit(50)
                ->get()
                ->mapWithKeys(fn ($employer) => [(string) $employer->_id => $employer->name])
                ->toArray();
        });
    }

      /**
     * Czyści cache opcji selectów
     */
    public static function clearSelectCache(): void
    {
        Cache::forget('employer_select_options');
    }

    protected static function booted()
    {
        static::saved(function () {
            static::clearSelectCache();
        });

        static::deleted(function () {
            static::clearSelectCache();
        });
    }
}
