<?php

namespace Tests\Unit;

use MbeGroup\HomeLayout\Models\HomeLayout;
use MbeGroup\HomeLayout\Resources\ArticlesHomeLayoutResource;
use MbeGroup\HomeLayout\Resources\HomeLayoutResource;
use MbeGroup\HomeLayout\Services\HomeLayoutService;
use MbeGroup\HomeLayout\Contracts\HomeLayoutServiceInterface;
use PHPUnit\Framework\TestCase;

class HomeLayoutServiceTest extends TestCase
{
    protected HomeLayoutService $service;

    protected function setUp(): void
    {
        parent::setUp();
        $this->service = new HomeLayoutService();
    }

    public function test_service_implements_interface(): void
    {
        $this->assertInstanceOf(HomeLayoutServiceInterface::class, $this->service);
    }

    public function test_get_resource_by_slug_returns_articles_resource_for_articles_slug(): void
    {
        $resourceClass = $this->service->getResourceBySlug('articles');

        $this->assertEquals(ArticlesHomeLayoutResource::class, $resourceClass);
    }

    public function test_get_resource_by_slug_returns_default_resource_for_unknown_slug(): void
    {
        $resourceClass = $this->service->getResourceBySlug('unknown-slug');

        $this->assertEquals(HomeLayoutResource::class, $resourceClass);
    }

    public function test_get_resource_by_slug_returns_default_resource_for_homepage_slug(): void
    {
        $resourceClass = $this->service->getResourceBySlug('homepage');

        $this->assertEquals(HomeLayoutResource::class, $resourceClass);
    }

    public function test_get_resource_by_slug_returns_default_resource_for_empty_slug(): void
    {
        $resourceClass = $this->service->getResourceBySlug('');

        $this->assertEquals(HomeLayoutResource::class, $resourceClass);
    }

    public function test_get_resource_by_slug_returns_default_resource_for_random_slug(): void
    {
        $resourceClass = $this->service->getResourceBySlug('random-layout-name');

        $this->assertEquals(HomeLayoutResource::class, $resourceClass);
    }

    public function test_get_resource_by_slug_returns_articles_resource_only_for_exact_articles_match(): void
    {
        // Test exact match
        $this->assertEquals(
            ArticlesHomeLayoutResource::class,
            $this->service->getResourceBySlug('articles')
        );

        // Test that similar strings don't match
        $this->assertEquals(
            HomeLayoutResource::class,
            $this->service->getResourceBySlug('article')
        );

        $this->assertEquals(
            HomeLayoutResource::class,
            $this->service->getResourceBySlug('articles-page')
        );

        $this->assertEquals(
            HomeLayoutResource::class,
            $this->service->getResourceBySlug('ARTICLES')
        );
    }

    public function test_get_resource_by_slug_handles_special_characters(): void
    {
        $resourceClass = $this->service->getResourceBySlug('special-!@#$%');

        $this->assertEquals(HomeLayoutResource::class, $resourceClass);
    }

    public function test_get_resource_by_slug_handles_numeric_slugs(): void
    {
        $resourceClass = $this->service->getResourceBySlug('12345');

        $this->assertEquals(HomeLayoutResource::class, $resourceClass);
    }

    public function test_service_has_get_method(): void
    {
        $this->assertTrue(method_exists($this->service, 'get'));
    }

    public function test_service_has_get_resource_by_slug_method(): void
    {
        $this->assertTrue(method_exists($this->service, 'getResourceBySlug'));
    }
}
