<?php

namespace Tests\Unit;

use MbeGroup\Homepage\Models\Homepage;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

class HomepageTest extends TestCase
{
    /**
     * Test tworzenia nowej instancji Homepage
     */
    public function test_can_create_homepage_instance(): void
    {
        $homepage = new Homepage();
        
        $this->assertInstanceOf(Homepage::class, $homepage);
    }

    /**
     * Test sprawdzający czy klasa ma odpowiednie właściwości
     */
    public function test_homepage_class_properties(): void
    {
        $reflection = new ReflectionClass(Homepage::class);
        
        // Sprawdzenie czy klasa dziedziczy po MongoDB\Laravel\Eloquent\Model
        $this->assertTrue($reflection->hasProperty('connection'));
        $this->assertTrue($reflection->hasProperty('collection'));
        $this->assertTrue($reflection->hasProperty('fillable'));
        $this->assertTrue($reflection->hasProperty('casts'));
    }

    /**
     * Test sprawdzający wypełnialne pola
     */
    public function test_fillable_attributes(): void
    {
        $homepage = new Homepage();
        $reflection = new ReflectionClass($homepage);
        
        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($homepage);
        
        $expectedFillable = ['slug', 'name', 'enabled', 'data'];
        $this->assertEquals($expectedFillable, $fillable);
    }

    /**
     * Test sprawdzający connection
     */
    public function test_mongodb_connection(): void
    {
        $homepage = new Homepage();
        $reflection = new ReflectionClass($homepage);
        
        $connectionProperty = $reflection->getProperty('connection');
        $connectionProperty->setAccessible(true);
        $connection = $connectionProperty->getValue($homepage);
        
        $this->assertEquals('mongodb', $connection);
    }

    /**
     * Test sprawdzający nazwę kolekcji
     */
    public function test_collection_name(): void
    {
        $homepage = new Homepage();
        $reflection = new ReflectionClass($homepage);
        
        $collectionProperty = $reflection->getProperty('collection');
        $collectionProperty->setAccessible(true);
        $collection = $collectionProperty->getValue($homepage);
        
        $this->assertEquals('homepages', $collection);
    }

    /**
     * Test sprawdzający casts
     */
    public function test_casts_configuration(): void
    {
        $homepage = new Homepage();
        $reflection = new ReflectionClass($homepage);
        
        $castsProperty = $reflection->getProperty('casts');
        $castsProperty->setAccessible(true);
        $casts = $castsProperty->getValue($homepage);
        
        $this->assertArrayHasKey('data', $casts);
        $this->assertEquals('array', $casts['data']);
    }

    /**
     * Test sprawdzający namespace klasy
     */
    public function test_class_namespace(): void
    {
        $homepage = new Homepage();
        $reflection = new ReflectionClass($homepage);
        
        $this->assertEquals('MbeGroup\Homepage\Models', $reflection->getNamespaceName());
        $this->assertEquals('Homepage', $reflection->getShortName());
    }

    /**
     * Test czy klasa implementuje odpowiednie metody (sprawdzenie czy dziedziczy po Model)
     */
    public function test_extends_mongodb_model(): void
    {
        $homepage = new Homepage();
        
        // Sprawdzenie czy klasa dziedziczy po MongoDB Laravel Eloquent Model
        $this->assertInstanceOf(\MongoDB\Laravel\Eloquent\Model::class, $homepage);
    }

    /**
     * Test konstruktora z danymi
     */
    public function test_constructor_with_attributes(): void
    {
        $attributes = [
            'slug' => 'test-homepage',
            'name' => 'Test Homepage',
            'enabled' => true,
            'data' => ['key' => 'value']
        ];

        // Test czy konstruktor przyjmuje tablicę atrybutów
        $homepage = new Homepage($attributes);
        $this->assertInstanceOf(Homepage::class, $homepage);
    }

    /**
     * Test sprawdzający czy wszystkie wymagane właściwości są zdefiniowane
     */
    public function test_all_required_properties_are_defined(): void
    {
        $homepage = new Homepage();
        $reflection = new ReflectionClass($homepage);
        
        $expectedProperties = ['connection', 'collection', 'fillable', 'casts'];
        
        foreach ($expectedProperties as $property) {
            $this->assertTrue(
                $reflection->hasProperty($property),
                "Właściwość {$property} nie została znaleziona w klasie Homepage"
            );
        }
    }

    /**
     * Celowy test, który ma nie przejść (dla demonstracji: jeden test przejdzie, drugi nie)
     */
    public function test_intentional_failure(): void
    {
        // Ten test celowo sprawdza fałsz, aby wymusić niepowodzenie
        $this->assertTrue(false, 'Celowe niepowodzenie testu');
    }
}