<?php

namespace MbeGroup\Job\Filament\Resources;

use MbeGroup\Job\Models\ArchivedJobOffer;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use MbeGroup\Job\Filament\Resources\ArchiveJobOfferResource\Pages\ListArchiveJobOffers;
use MbeGroup\Job\Filament\Resources\ArchiveJobOfferResource\Pages\EditArchiveJobOffer;
use Filament\Actions;
use Filament\Forms;
use App\ValueObject\Status;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use Illuminate\Support\Facades\Log;
use App\Helpers\DictionaryHelper;
use Tapp\FilamentGoogleAutocomplete\Forms\Components\GoogleAutocomplete;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use MbeGroup\Employer\Models\Employer;
use App\Filament\Components\JobOfferTemplateFormComponents;
use MbeGroup\Job\Contracts\JobOfferArchiveServiceInterface;;

class ArchiveJobOfferResource extends Resource
{
    protected static ?string $model = ArchivedJobOffer::class;

          /**
     * Configure Status field with common hydration and dehydration logic
     */
    private static function configureStatusField(Forms\Components\Select $field): Forms\Components\Select
    {
        return $field
            ->afterStateHydrated(function (Forms\Components\Select $component, $state): void {
                if ($state instanceof Status) {
                    $component->state($state->value);
                }
            })
            ->dehydrateStateUsing(function ($state): int {
                return $state instanceof Status ? $state->value: (int) $state;
            })
            ->rule('integer');
    }

    public static function getNavigationGroup(): string
    {
        return __('job::filament.navigation.group.label');
    }

    public static function getNavigationLabel(): string
    {
        return __('job::filament.resources.archive_job_offer.navigation_label');
    }

    public static function getModelLabel(): string
    {
        return __('job::filament.resources.archive_job_offer.model_label');
    }

    public static function getPluralModelLabel(): string
    {
        return __('job::filament.resources.archive_job_offer.plural_model_label');
    }

    public static function form(Schema $schema): Schema
    {
        $statusOptions     = Status::getJobOfferLabels();
        $dictionaryOptions = [
            'industry'                    => DictionaryHelper::getHierarchicalNamesByType('industry'),
            'specialization'              => DictionaryHelper::getHierarchicalNamesByType('specialization'),
            'position_level'              => DictionaryHelper::getHierarchicalNamesByType('position_level'),
            'operating_mode'              => DictionaryHelper::getHierarchicalNamesByType('operating_mode'),
            'hourly_rate'                 => DictionaryHelper::getHierarchicalNamesByType('hourly_rate'),
            'employment_type'             => DictionaryHelper::getHierarchicalNamesByType('employment_type'),
            'application_method'          => DictionaryHelper::getSlugsForOptionsByType('application_method'),
            'preferred_study'             => DictionaryHelper::getHierarchicalNamesByType('preferred_study'),
            'technologies_and_tools'      => DictionaryHelper::getHierarchicalNamesByType('technologies_and_tools'),
            'skills_and_methodologies'    => DictionaryHelper::getHierarchicalNamesByType('skills_and_methodologies'),
            'licenses_and_qualifications' => DictionaryHelper::getHierarchicalNamesByType('licenses_and_qualifications'),
            'languages'                   => DictionaryHelper::getHierarchicalNamesByType('languages'),
            'salary_type'                 => DictionaryHelper::getHierarchicalNamesByType('salary_type'),
            'salary_form'                 => DictionaryHelper::getHierarchicalNamesByType('salary_form'),
            'salary_currency'             => DictionaryHelper::getHierarchicalNamesByType('currency'),
            'position'                    => DictionaryHelper::getHierarchicalNamesByType('position'),
        ];

        return $schema
            ->schema([
                Tabs::make('Tabs')
                    ->persistTabInQueryString()
                    ->tabs([
                        Tab::make(__('job::filament.resources.job_offer.form.tabs.basic_info'))
                            ->schema([
                                Forms\Components\Select::make('employer_id')
                                    ->label(__('content::filament.resources.article.form.fields.employer_id'))
                                    ->options(Employer::getSelectOptions())
                                    ->searchable()
                                    ->required()
                                    ->getSearchResultsUsing(fn (string $search) => Employer::searchForSelect($search)),
                                Forms\Components\Select::make('product_statistics')
                                    ->label(__('job::filament.resources.job_offer.form.fields.product_statistics_id'))
                                    ->options(function (callable $get) {
                                        $employerId = $get('employer_id');
                                        if (!$employerId) {
                                            return [];
                                        }
                                        try {
                                            $products = \MbeGroup\Product\Models\ProductStatistics::where('employer_id', $employerId)
                                                ->where('status', Status::ACTIVE)
                                                ->where('quantity', '>', 0)
                                                ->with(['product' => function ($query) {
                                                    $query->where('category', 'product-job-offers');
                                                }])
                                                ->get()
                                                ->mapWithKeys(fn ($stat) => [
                                                    (string) $stat->id => ($stat->product?->name ?? 'N/A') . ' (' . $stat->created_at?->format('Y-m-d H:i') . ')'
                                                ])
                                                ->toArray();

                                            return $products;
                                        } catch (\Exception $e) {
                                            Log::error('Error fetching products: ' . $e->getMessage());
                                            return [];
                                        }
                                    })
                                    ->searchable()
                                    ->live()
                                    ->afterStateUpdated(function ($state, callable $set) {
                                        if (!$state) {
                                            $set('expiration_date', now()->addMonths(1));
                                            return;
                                        }
                                        try {
                                            $productStatistics = \MbeGroup\Product\Models\ProductStatistics::find($state);
                                            if ($productStatistics && $productStatistics->product) {
                                                $months = $productStatistics->product->product_life_months ?? 1;
                                                $set('expiration_date', now()->addMonths($months));
                                            }
                                        } catch (\Exception $e) {
                                            Log::error('Error updating expiration_date from product_statistics: ' . $e->getMessage());
                                            $set('expiration_date', now()->addMonths(1));
                                        }
                                    })
                                    ->required(fn (callable $get) => $get('status') === Status::ACTIVE),
                                Forms\Components\TextInput::make('title')
                                    ->label(__('job::filament.resources.job_offer.form.fields.title'))
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (string $operation, $state, callable $set) {
                                        if ($operation !== 'create') {
                                            return;
                                        }
                                        $set('slug', \Illuminate\Support\Str::slug($state));
                                    }),
                                Forms\Components\TextInput::make('slug')
                                    ->label(__('job::filament.resources.job_offer.form.fields.slug'))
                                    ->maxLength(255)
                                    ->disabled(fn (string $operation): bool => $operation === 'edit')
                                    ->dehydrated(fn (string $operation): bool => $operation === 'create'),
                                Forms\Components\TextInput::make('reference_number')
                                    ->label(__('job::filament.resources.job_offer.form.fields.reference_number')),
                                Forms\Components\Select::make('application_method.apply_options.application_method')
                                    ->label(__('job::filament.resources.job_offer.form.fields.application_method'))
                                    ->options($dictionaryOptions['application_method'])
                                    ->allowHtml()
                                    ->searchable()
                                    ->required()
                                    ->live()
                                    ->afterStateUpdated(function ($state, callable $set) {
                                        if ($state !== 'email') {
                                            $set('application_method.apply_options.application_email', null);
                                        }
                                        if ($state !== 'phone') {
                                            $set('application_method.apply_options.application_phone', null);
                                        }
                                        if ($state !== 'url') {
                                            $set('application_method.apply_options.application_url', null);
                                        }
                                    }),
                                Forms\Components\TextInput::make('application_method.apply_options.application_link_reference')
                                    ->label(__('job::filament.resources.job_offer.form.fields.application_link_reference'))
                                    ->url()
                                    ->required()
                                    ->visible(fn (callable $get) => $get('application_method.apply_options.application_method') === 'employer-link'),
                                Forms\Components\TextInput::make('application_method.apply_options.application_link_reference_value')
                                    ->label(__('job::filament.resources.job_offer.form.fields.application_link_reference_value'))
                                    ->visible(fn (callable $get) => $get('application_method.apply_options.application_method') === 'employer-link'),
                                Forms\Components\TextInput::make('application_method.apply_options.application_email')
                                    ->label(__('job::filament.resources.job_offer.form.fields.application_email'))
                                    ->email()
                                    ->required()
                                    ->visible(fn (callable $get) => $get('application_method.apply_options.application_method') === 'kwf-form')
                                    ->maxLength(255),
                                Forms\Components\Checkbox::make('application_method.apply_options.application_remote_recruitment')
                                    ->label(__('job::filament.resources.job_offer.form.fields.application_remote_recruitment'))
                                    ->visible(fn (callable $get) => $get('application_method.apply_options.application_method') === 'kwf-form')
                                    ->default(false),
                                ]),
                    Tab::make(__('job::filament.resources.job_offer.form.tabs.category'))
                        ->schema([
                            Forms\Components\Repeater::make('locations')
                                ->label(__('job::filament.resources.job_offer.form.fields.locations'))
                                ->lazy()
                                ->required()
                                ->schema([
                                    GoogleAutocomplete::make('google_search')
                                        ->label(__('job::filament.resources.job_offer.form.fields.google_search'))
                                        ->countries([
                                            'PL',
                                        ])
                                        ->autocompletePlaceholder('Wyszukaj lokalizację')
                                        ->language('pl')
                                        ->lazy()
                                        ->withFields([
                                            Forms\Components\TextInput::make('location_name')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{locality}',
                                                ])
                                                ->label(__('job::filament.resources.job_offer.form.fields.location_name')),
                                            Forms\Components\TextInput::make('municipality')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{administrative_area_level_2}',
                                                ])
                                                ->label(__('job::filament.resources.job_offer.form.fields.municipality')),
                                            Forms\Components\TextInput::make('region')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{administrative_area_level_1}',
                                                ])
                                                ->label(__('job::filament.resources.job_offer.form.fields.region')),
                                            Forms\Components\TextInput::make('postal_code')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{postal_code}',
                                                ])
                                                ->label(__('job::filament.resources.job_offer.form.fields.postal_code')),
                                            Forms\Components\TextInput::make('street')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{route}',
                                                ])
                                                ->label(__('job::filament.resources.job_offer.form.fields.street')),
                                            Forms\Components\TextInput::make('building_number')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{street_number}',
                                                ])
                                                ->label(__('job::filament.resources.job_offer.form.fields.building_number')),
                                            Forms\Components\TextInput::make('latitude')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{latitude}',
                                                ])
                                                ->label(__('job::filament.resources.job_offer.form.fields.latitude')),
                                            Forms\Components\TextInput::make('longitude')
                                                ->extraInputAttributes([
                                                    'data-google-field' => '{longitude}',
                                                ])
                                                ->label(__('job::filament.resources.job_offer.form.fields.longitude')),
                                            Grid::make(3)
                                                ->schema([
                                                    Forms\Components\TextInput::make('phone')
                                                        ->label(__('job::filament.resources.job_offer.form.fields.phone'))
                                                        ->tel()
                                                        ->maxLength(20),
                                                    Forms\Components\TextInput::make('email')
                                                        ->label(__('job::filament.resources.job_offer.form.fields.email'))
                                                        ->email()
                                                        ->maxLength(255),
                                                    Forms\Components\TextInput::make('website')
                                                        ->label(__('job::filament.resources.job_offer.form.fields.website'))
                                                        ->url()
                                                        ->maxLength(255),
                                                ]),
                                            self::configureStatusField(
                                                Forms\Components\Select::make('status')
                                                    ->label(__('job::filament.resources.job_offer.form.fields.status'))
                                                    ->options($statusOptions)
                                                    ->required()
                                                    ->default(Status::ACTIVE)
                                            ),
                                            Forms\Components\Checkbox::make('location_is_default')
                                                ->label(__('job::filament.resources.job_offer.form.fields.location_is_default'))
                                                ->live()
                                                ->afterStateUpdated(function ($state, callable $get, callable $set) {
                                                    if ($state) {
                                                        $allLocations = $get('../../locations') ?? [];
                                                        foreach ($allLocations as $index => $location) {
                                                            $set("../../locations.{$index}.location_is_default", false);
                                                        }
                                                        $set('location_is_default', true);
                                                    }
                                                }),
                                            ])
                                ])
                                ->columns(3)
                                ->defaultItems(0)
                                ->addActionLabel(__('job::filament.resources.job_offer.form.fields.add_location'))
                                ->extraAttributes(['class' => 'add-location-btn'])
                                ->reorderable(false)
                                ->collapsible()
                                ->collapsed(true)
                                ->cloneable(false)
                                ->itemLabel(function (array $state): ?string {
                                    if (empty($state['location_name'])) {
                                        return null;
                                    }
                                    return $state['location_name'] . (isset($state['region']) ? ' (' . $state['region'] . ')' : '');
                                }),
                            Forms\Components\Select::make('industry_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.industry'))
                                ->options($dictionaryOptions['industry'])
                                ->allowHtml()
                                ->searchable()
                                ->required(),
                            Forms\Components\Select::make('specialization_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.specialization'))
                                ->options($dictionaryOptions['specialization'])
                                ->allowHtml()
                                ->searchable()
                                ->required(),
                            Forms\Components\Select::make('position_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.position'))
                                ->options($dictionaryOptions['position'])
                                ->allowHtml()
                                ->searchable(),
                            Forms\Components\Select::make('position_level_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.position_level'))
                                ->options($dictionaryOptions['position_level'])
                                ->allowHtml()
                                ->searchable(),
                            Forms\Components\Checkbox::make('is_for_technicians')
                                ->label(__('job::filament.resources.job_offer.form.fields.is_for_technicians')),
                            Forms\Components\Select::make('operating_mode_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.operating_mode'))
                                ->options($dictionaryOptions['operating_mode'])
                                ->allowHtml()
                                ->searchable(),
                            Forms\Components\Checkbox::make('is_remote_recruitment')
                                ->label(__('job::filament.resources.job_offer.form.fields.is_remote_recruitment')),
                            Forms\Components\Select::make('hourly_rate_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.hourly_rate'))
                                ->options($dictionaryOptions['hourly_rate'])
                                ->allowHtml()
                                ->searchable(),
                            Section::make(__('job::filament.resources.job_offer.form.sections.requirements'))
                                ->schema([
                                    Forms\Components\Select::make('preferred_study_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.preferred_study'))
                                        ->options($dictionaryOptions['preferred_study'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                    Forms\Components\Select::make('language_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.languages'))
                                        ->options($dictionaryOptions['languages'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                    Forms\Components\Select::make('technologies_and_tools_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.technologies_and_tools'))
                                        ->options($dictionaryOptions['technologies_and_tools'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                    Forms\Components\Select::make('skills_and_methodologies_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.skills_and_methodologies'))
                                        ->options($dictionaryOptions['skills_and_methodologies'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                    Forms\Components\Select::make('licenses_and_qualifications_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.licenses_and_qualifications'))
                                        ->options($dictionaryOptions['licenses_and_qualifications'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                ]),
                            Section::make(__('job::filament.resources.job_offer.form.sections.nice_to_have'))
                                ->schema([
                                    Forms\Components\Select::make('nice_to_have_preferred_study_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.preferred_study'))
                                        ->options($dictionaryOptions['preferred_study'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                    Forms\Components\Select::make('nice_to_have_language_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.languages'))
                                        ->options($dictionaryOptions['languages'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                    Forms\Components\Select::make('nice_to_have_technologies_and_tools_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.technologies_and_tools'))
                                        ->options($dictionaryOptions['technologies_and_tools'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                    Forms\Components\Select::make('nice_to_have_skills_and_methodologies_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.skills_and_methodologies'))
                                        ->options($dictionaryOptions['skills_and_methodologies'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                    Forms\Components\Select::make('nice_to_have_licenses_and_qualifications_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.licenses_and_qualifications'))
                                        ->options($dictionaryOptions['licenses_and_qualifications'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                ]),
                        ]),
                    Tab::make(__('job::filament.resources.job_offer.form.tabs.salary_and_contracts'))
                        ->schema([
                            Forms\Components\Repeater::make('salary')
                            ->label(__('job::filament.resources.job_offer.form.fields.contracts'))
                            ->lazy()
                            ->addActionLabel(__('job::filament.resources.job_offer.form.fields.add_contract'))
                            ->required()
                            ->schema([
                                Forms\Components\Select::make('salary_employment_type_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.salary_employment_type'))
                                ->options($dictionaryOptions['employment_type'])
                                ->allowHtml()
                                ->searchable(),
                                Section::make(__('job::filament.resources.job_offer.form.fields.salary'))
                                    ->schema([
                                        Forms\Components\Select::make('salary_type')
                                            ->label(__('job::filament.resources.job_offer.form.fields.salary_type'))
                                            ->options($dictionaryOptions['salary_type'])
                                            ->allowHtml(),
                                        Forms\Components\Select::make('salary_form')
                                            ->label(__('job::filament.resources.job_offer.form.fields.salary_form'))
                                            ->options($dictionaryOptions['salary_form'])
                                            ->allowHtml(),
                                        Forms\Components\TextInput::make('salary_from')
                                            ->label(__('job::filament.resources.job_offer.form.fields.salary_from'))
                                            ->numeric(),
                                        Forms\Components\TextInput::make('salary_to')
                                            ->label(__('job::filament.resources.job_offer.form.fields.salary_to'))
                                            ->numeric(),
                                        Forms\Components\Select::make('salary_currency')
                                            ->label(__('job::filament.resources.job_offer.form.fields.salary_currency'))
                                            ->options($dictionaryOptions['salary_currency'])
                                            ->allowHtml(),
                                        Forms\Components\Checkbox::make('salary_range_is_visible')
                                            ->label(__('job::filament.resources.job_offer.form.fields.salary_range_is_visible'))
                                            ->default(true),
                                    ])
                                    ->collapsed()
                                    ->collapsible(),
                            ]),
                            Section::make(__('job::filament.resources.job_offer.form.fields.salary_for_search'))
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            Forms\Components\TextInput::make('salary_for_search.from')
                                                ->label(__('job::filament.resources.job_offer.form.fields.salary_for_search_from'))
                                                ->disabled(),
                                            Forms\Components\TextInput::make('salary_for_search.to')
                                                ->label(__('job::filament.resources.job_offer.form.fields.salary_for_search_to'))
                                                ->disabled(),
                                        ]),
                                ]),
                        ]),
                    Tab::make(__('job::filament.resources.job_offer.form.tabs.content'))
                        ->schema(
                            function (callable $get) {
                                return JobOfferTemplateFormComponents::getJobOfferTemplate($get);
                            }),
                    Tab::make(__('job::filament.resources.job_offer.form.tabs.publication_settings'))
                        ->schema([
                            Forms\Components\Select::make('status')
                                ->label(__('job::filament.resources.job_offer.form.fields.status'))
                                ->options($statusOptions)
                                ->required()
                                ->live()
                                ->default(Status::DRAFT),
                            Forms\Components\DateTimePicker::make('created_at')
                                ->label(__('job::filament.resources.job_offer.form.fields.created_at'))
                                ->default(now())
                                ->required(),
                            Forms\Components\DateTimePicker::make('updated_at')
                                ->label(__('job::filament.resources.job_offer.form.fields.updated_at'))
                                ->default(fn (string $operation) => $operation === 'create' ? now() : null),
                            Forms\Components\DateTimePicker::make('publication_date')
                                ->label(__('job::filament.resources.job_offer.form.fields.publication_date'))
                                ->default(now()),
                            Forms\Components\DateTimePicker::make('expiration_date')
                                ->label(__('job::filament.resources.job_offer.form.fields.expiration_date'))
                                ->default(function (callable $get) {
                                    $productStatisticsId = $get('product_statistics');
                                    if (!$productStatisticsId) {
                                        return now()->addMonths(1);
                                    }
                                    try {
                                        $productStatistics = \MbeGroup\Product\Models\ProductStatistics::find($productStatisticsId);
                                        if ($productStatistics && $productStatistics->product) {
                                            $months = $productStatistics->product->product_life_months ?? 1;
                                            return now()->addMonths($months);
                                        }
                                    } catch (\Exception $e) {
                                        Log::error('Error fetching product_life_months: ' . $e->getMessage());
                                    }
                                    return now()->addMonths(1);
                                })
                                ->readOnly(true),
                            Forms\Components\Checkbox::make('is_top_offer')
                                ->label(__('job::filament.resources.job_offer.form.fields.is_top_offer')),
                            Forms\Components\Select::make('refresh_frequency')
                                ->label(__('job::filament.resources.job_offer.form.fields.refresh_frequency'))
                                ->options([
                                    1 => 1, 7 => 7, 14 => 14
                                ])
                                ->default('none')
                        ])

                    ]),
            ])->columns(1);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label('ID')
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->sortable(),
                Tables\Columns\TextColumn::make('title')
                    ->label(__('job::filament.resources.job_offer.table.columns.title'))
                    ->searchable()
                    ->toggleable()
                    ->limit(50),
                Tables\Columns\TextColumn::make('employer.name')
                    ->label(__('content::filament.resources.content.table.columns.employer'))
                    ->placeholder('Brak danych')
                    ->searchable()
                    ->toggleable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('locations')
                    ->label(__('job::filament.resources.job_offer.table.columns.location'))
                    ->formatStateUsing(function ($state) {
                        if (empty($state) || !is_array($state)) {
                            return 'Brak danych';
                        }
                        return $state['location_name'];
                    })
                    ->toggleable()
                    ->searchable(),
                Tables\Columns\IconColumn::make('status')
                    ->label(__('job::filament.resources.job_offer.table.columns.status'))
                    ->state(fn ($record) => Status::int($record->status))
                    ->icon(fn ($state) => Status::icon($state))
                    ->color(fn ($state) => Status::color($state))
                    ->tooltip(fn ($state) => Status::label($state))
                    ->toggleable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label(__('job::filament.resources.job_offer.table.columns.created_at'))
                    ->dateTime()
                    ->toggleable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('publication_date')
                    ->label(__('job::filament.resources.job_offer.table.columns.publication_date'))
                    ->dateTime()
                    ->toggleable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('expiration_date')
                    ->label(__('job::filament.resources.job_offer.table.columns.expiration_date'))
                    ->dateTime()
                    ->toggleable()
                    ->sortable(),
            ])
            ->filters([
                    //
            ])
            ->recordActions([
                Actions\EditAction::make()
                    ->iconButton()
                    ->color('info'),
                Actions\ViewAction::make()
                    ->iconButton()
                    ->color('gray'),
                Actions\Action::make('restore')
                    ->iconButton()
                    ->icon('heroicon-o-arrow-uturn-left')
                    ->label(__('job::filament.resources.archive_job_offer.table.actions.restore.label'))
                    ->modalHeading(__('job::filament.resources.archive_job_offer.table.actions.restore.modal.heading'))
                    ->requiresConfirmation()
                    ->color('success')
                    ->modalDescription(__('job::filament.resources.archive_job_offer.table.actions.restore.modal.description'))
                    ->action(function (ArchivedJobOffer $record) {
                        app(JobOfferArchiveServiceInterface::class)->restoreOffer($record);
                    }),
                Actions\DeleteAction::make()
                    ->iconButton()
                    ->color('danger'),
            ])
            ->toolbarActions([
                Actions\BulkActionGroup::make([
                    Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->description('Version: ' . \Composer\InstalledVersions::getVersion('mbegroup/job'));
    }

    public static function getRelations(): array
    {
        return [
                  //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index'  => ListArchiveJobOffers::route('/'),
            'edit'   => EditArchiveJobOffer::route('/{record}/edit'),
        ];
    }
}
