<?php
namespace MbeGroup\Job\Models;

use MongoDB\Laravel\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;
use MbeGroup\Employer\Models\Employer;
use App\ValueObject\Status;

class Import extends Model
{
    protected $connection = 'mongodb';
    protected $collection = 'imports';

    protected $fillable = [
        'name',
        'provider_type',
        'url',
        'employer_id',
        'template_name',
        'scheduled_time',
        'status',
        'config',
        'last_run_at',
        'runs_history',
        'is_running',
    ];

    protected $casts = [
        'status'           => Status::class,
        'last_run_at'  => 'datetime',
        'runs_history' => 'array',
        'config'       => 'array',
        'is_running'   => 'boolean',
    ];

    public function employer()
    {
        return $this->belongsTo(Employer::class, 'employer_id');
    }

    public function scopeShouldRun($query)
    {
        $currentTime = now()->format('H:i:s');
        $today       = now()->toDateString();

        return $query
            ->where('status', Status::ACTIVE)
            ->where(function ($q) use ($today) {
                $q->whereNull('last_run_at')
                  ->orWhere('last_run_at', '<', Carbon::parse($today)->startOfDay());
            })
            ->where('scheduled_time', '<=', $currentTime);
    }

    public function addRunToHistory(array $runData): void
    {
        $history = $this->runs_history ?? [];

        $history[] = array_merge($runData, [
            'created_at' => now()->toISOString()
        ]);

        if (count($history) > 100) {
            $history = array_slice($history, -100);
        }

        $this->runs_history = $history;
        $this->save();
    }

    public function getLastRun(): ?array
    {
        $history = $this->runs_history ?? [];
        return !empty($history) ? end($history) : null;
    }

    public function getSuccessfulRunsCount(): int
    {
        $history = $this->runs_history ?? [];
        return count(array_filter($history, fn($run) => ($run['status'] ?? '') === 'success'));
    }

    public function getFailedRunsCount(): int
    {
        $history = $this->runs_history ?? [];
        return count(array_filter($history, fn($run) => ($run['status'] ?? '') === 'failed'));
    }

    public function getScheduledTimeAttribute($value)
    {
        return $value;
    }

    public function setScheduledTimeAttribute($value)
    {
        if ($value instanceof \DateTime || $value instanceof Carbon) {
            $this->attributes['scheduled_time'] = $value->format('H:i:s');
        } else {
            $this->attributes['scheduled_time'] = $value;
        }
    }
}
