<?php

namespace MbeGroup\Job\Services;

use MbeGroup\Job\Models\JobOffer;
use MbeGroup\Job\Models\ArchivedJobOffer;
use MbeGroup\Job\Contracts\JobOfferArchiveServiceInterface;
use App\ValueObject\Status;
use Illuminate\Support\Facades\DB;

class JobOfferArchiveService implements JobOfferArchiveServiceInterface
{
    public function archiveOffer(JobOffer $jobOffer, ?string $reason = null): ArchivedJobOffer
    {
        return DB::transaction(function () use ($jobOffer, $reason) {
            // Pobierz wszystkie atrybuty (włącznie z _id)
            $attributes = $jobOffer->getAttributes();

            // Usuń z indeksu wyszukiwania przed archiwizacją
            if ($jobOffer->shouldBeSearchable()) {
                $jobOffer->unsearchable();
            }

            // Usuń powiązanie z product_statistics
            unset($attributes['product_statistics']);

            // Dodaj metadane archiwizacji
            $attributes['archived_at'] = now();
            $attributes['archived_reason'] = $reason;

            // Utwórz zarchiwizowaną ofertę zachowując oryginalne _id
            $archivedOffer = new ArchivedJobOffer();
            $archivedOffer->setRawAttributes($attributes);
            $archivedOffer->save();

            // Usuń oryginalną ofertę
            $jobOffer->delete();

            return $archivedOffer;
        });
    }

    public function archiveOffers(array $offerIds, ?string $reason = null): array
    {
        return DB::transaction(function () use ($offerIds, $reason) {
            $archived = [];

            // Pobierz wszystkie oferty jednym zapytaniem (optymalizacja)
            $jobOffers = JobOffer::whereIn('_id', $offerIds)->get();

            foreach ($jobOffers as $jobOffer) {
                $archived[] = $this->archiveOffer($jobOffer, $reason);
            }

            return $archived;
        });
    }

    public function restoreOffer(ArchivedJobOffer $archivedOffer): JobOffer
    {
        return DB::transaction(function () use ($archivedOffer) {
            // Pobierz wszystkie atrybuty
            $attributes = $archivedOffer->getAttributes();

            // Usuń metadane archiwizacji
            unset($attributes['archived_at'], $attributes['archived_reason']);

            // Przy przywracaniu product_statistics nie będzie (było usunięte przy archiwizacji)
            // więc nie ma potrzeby go tutaj usuwać

            // Ustaw status na DRAFT przy przywracaniu
            $attributes['status'] = Status::DRAFT;

            // Utwórz ofertę zachowując oryginalne _id
            $jobOffer = new JobOffer();
            $jobOffer->setRawAttributes($attributes);
            $jobOffer->save();

            // Dodaj do indeksu wyszukiwania jeśli powinna być wyszukiwalna
            if ($jobOffer->shouldBeSearchable()) {
                $jobOffer->searchable();
            }

            // Usuń z archiwum
            $archivedOffer->delete();

            return $jobOffer;
        });
    }

    /**
     * Sprawdź czy oferta może być zarchiwizowana
     */
    public function canArchive(JobOffer $jobOffer): bool
    {
        // Możesz dodać własną logikę walidacji
        return !in_array($jobOffer->status?->value ?? $jobOffer->status, [
            Status::ARCHIVED, // jeśli masz taki status
        ]);
    }

    /**
     * Zarchiwizuj wygasłe oferty
     */
    public function archiveExpiredOffers(?string $reason = 'expired'): int
    {
        $expiredOffers = JobOffer::where('expiration_date', '<', now())
            ->where('status', Status::ACTIVE)
            ->get();

        $count = 0;
        foreach ($expiredOffers as $offer) {
            $this->archiveOffer($offer, $reason);
            $count++;
        }

        return $count;
    }

    /**
     * Pobierz statystyki archiwum
     */
    public function getArchiveStatistics(): array
    {
        return [
            'total' => ArchivedJobOffer::count(),
            'by_reason' => ArchivedJobOffer::raw(function ($collection) {
                return $collection->aggregate([
                    [
                        '$group' => [
                            '_id' => '$archived_reason',
                            'count' => ['$sum' => 1]
                        ]
                    ]
                ]);
            })->toArray(),
            'last_30_days' => ArchivedJobOffer::where('archived_at', '>=', now()->subDays(30))->count(),
        ];
    }
}
