<?php

namespace MbeGroup\Job\Services;

use MbeGroup\Job\Contracts\JobOffersSearchServiceInterface;
use Elastic\Elasticsearch\Client;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Pagination\LengthAwarePaginator as ConcreteLengthAwarePaginator;
use App\Support\FilterNormalizer;

class JobOffersSearchService implements JobOffersSearchServiceInterface
{
    protected Client $client;
    protected string $index = 'job_offers';

    public function __construct(Client $client)
    {
        $this->client = $client;
    }

    public function search(array $filters, int $perPage, int $page): LengthAwarePaginator
    {
        $esQuery = $this->buildElasticsearchQuery($filters, $perPage, $page);

        $response = $this->client->search($esQuery);
        $results = $response->asArray();

        $hits = $results['hits']['hits'] ?? [];
        $total = $results['hits']['total']['value'] ?? 0;

        $processedData = $this->processResults($hits);
        return new ConcreteLengthAwarePaginator(
            $processedData,
            $total,
            $perPage,
            $page,
            [
                'path' => request()->url(),
                'query' => request()->query(),
            ]
        );
    }

    private function buildElasticsearchQuery(array $filters, int $perPage, int $page): array
    {
        $query = [
            'index' => $this->index,
            'body'  => [
                'from' => ($page - 1) * $perPage,
                'size' => $perPage,
                'query' => [
                    'bool' => [
                        'must' => [],
                        'filter' => [],
                    ],
                ],
                'sort' => $this->buildSort($filters),
            ],
        ];

        if (!empty($filters['q'])) {
            $query['body']['query']['bool']['must'][] = [
                'multi_match' => [
                    'query' => $filters['q'],
                    'fields' => ['title^3', 'description', 'requirements'],
                ],
            ];
        }

        if ($filters['industry_ids'] ?? null) {
            $industryIds = FilterNormalizer::toIntArray($filters['industry_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['industry_id' => $industryIds]
            ];
        }

        if ($filters['specialization_ids'] ?? null) {
            $specializationIds = FilterNormalizer::toIntArray($filters['specialization_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['specialization_id' => $specializationIds]
            ];
        }

        if ($filters['employer_ids'] ?? null) {
            $employerIds = FilterNormalizer::toIntArray($filters['employer_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['employer.id' => $employerIds]
            ];
        }

        if (!empty($filters['lat']) && !empty($filters['lon'])) {
            $lat = (float) $filters['lat'];
            $lon = (float) $filters['lon'];
            $distance = (int) ($filters['distance'] ?? 50);

            $query['body']['query']['bool']['filter'][] = [
                'nested' => [
                    'path' => 'locations',
                    'query' => [
                        'geo_distance' => [
                            'distance' => "{$distance}km",
                            'locations.location' => [
                                'lat' => $lat,
                                'lon' => $lon
                            ]
                        ]
                    ]
                ]
            ];
        }

        if ($filters['position_ids'] ?? null) {
            $positionIds = FilterNormalizer::toIntArray($filters['position_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['position_id' => $positionIds]
            ];
        }

        if ($filters['position_level_ids'] ?? null) {
            $positionLevelIds = FilterNormalizer::toIntArray($filters['position_level_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['position_level_id' => $positionLevelIds]
            ];
        }

        if ($filters['operating_mode_ids'] ?? null) {
            $operatingModeIds = FilterNormalizer::toIntArray($filters['operating_mode_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['operating_mode_id' => $operatingModeIds]
            ];
        }

        if ($filters['salary_employment_type_ids'] ?? null) {
            $employmentTypeIds = FilterNormalizer::toIntArray($filters['salary_employment_type_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['employment_type_id' => $employmentTypeIds]
            ];
        }

        if ($filters['hourly_rate_ids'] ?? null) {
            $hourlyRateIds = FilterNormalizer::toIntArray($filters['hourly_rate_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['hourly_rate_id' => $hourlyRateIds]
            ];
        }

        if (!empty($filters['only_with_salary']) && $filters['only_with_salary']) {
            $query['body']['query']['bool']['filter'][] = [
                'exists' => ['field' => 'salary_for_search']
            ];
        }

        if (!empty($filters['min_salary'])) {
            $minSalary = (int) $filters['min_salary'];
            $query['body']['query']['bool']['filter'][] = [
                'range' => [
                    'salary_for_search.to' => [
                    'gte' => $minSalary
                    ]
                ]
            ];
        }

        if ($filters['preferred_study_ids'] ?? null) {
            $preferredStudyIds = FilterNormalizer::toIntArray($filters['preferred_study_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['preferred_study_ids' => $preferredStudyIds]
            ];
        }

        if ($filters['language_ids'] ?? null) {
            $languageIds = FilterNormalizer::toIntArray($filters['language_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['language_ids' => $languageIds]
            ];
        }

        if ($filters['technologies_and_tools_ids'] ?? null) {
            $technologyIds = FilterNormalizer::toIntArray($filters['technologies_and_tools_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['technologies_and_tools_ids' => $technologyIds]
            ];
        }

        if ($filters['skills_and_methodologies_ids'] ?? null) {
            $skillIds = FilterNormalizer::toIntArray($filters['skills_and_methodologies_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['skills_and_methodologies_ids' => $skillIds]
            ];
        }

        if ($filters['licenses_and_qualifications_ids'] ?? null) {
            $licenseIds = FilterNormalizer::toIntArray($filters['licenses_and_qualifications_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['licenses_and_qualifications_ids' => $licenseIds]
            ];
        }

        if (!empty($filters['is_remote_recruitment'])) {
            $isRemote = (bool) $filters['is_remote_recruitment'];
            $query['body']['query']['bool']['filter'][] = [
                'term' => ['is_remote_recruitment' => $isRemote]
            ];
        }

        if (!empty($filters['is_for_technicians'])) {
            $isForTechnicians = (bool) $filters['is_for_technicians'];
            $query['body']['query']['bool']['filter'][] = [
                'term' => ['is_for_technicians' => $isForTechnicians]
            ];
        }

        if ($filters['company_size_ids'] ?? null) {
            $companySizeIds = FilterNormalizer::toIntArray($filters['company_size_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['employer.company_size_id' => $companySizeIds]
            ];
        }

        if ($filters['company_type_ids'] ?? null) {
            $companyTypeIds = FilterNormalizer::toIntArray($filters['company_type_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['employer.company_type_id' => $companyTypeIds]
            ];
        }

        return $query;
    }

    private function processResults(array $hits): array
    {
        $results = [];
        foreach ($hits as $hit) {
            $results[] = $hit['_source'];
        }
        return $results;
    }

    private function buildSort(array $filters): array
    {
        $sortBy = $filters['sort_by'] ?? 'newest';

        return match ($sortBy) {
            'oldest' => [
                ['created_at' => ['order' => 'asc']]
            ],
            'newest' => [
                ['created_at' => ['order' => 'desc']]
            ],
            default => [
                ['created_at' => ['order' => 'desc']]
            ]
        };
    }
}
