<?php

namespace MbeGroup\Job\Controllers\Api\V1;

use App\Http\Controllers\Api\V1\CrudApiController;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use MbeGroup\Job\Resources\ApplicationResource;
use MbeGroup\Job\Contracts\ApplicationServiceInterface;
use MbeGroup\Job\Requests\StoreApplicationRequest;
use MbeGroup\Job\Models\Application;
use Illuminate\Support\Facades\Storage;

final class ApplicationController extends CrudApiController
{
    protected string $resourceName = 'application';

    public function __construct(
        protected ApplicationServiceInterface $searchService,
    ) {
    }

    public function index(Request $request): JsonResponse
    {
        return $this->notImplemented('index');
    }

    public function show(string $id): JsonResponse
    {
        return $this->notImplemented('show');
    }

    public function store(Request $request): JsonResponse
    {
        $formRequest = app(StoreApplicationRequest::class);
        $validated   = $formRequest->validated();

        $jobOffer        = $formRequest->getJobOffer();
        $applicationType = $formRequest->getApplicationType();

        try {
            $application = Application::create([
                'job_offer_id'         => $jobOffer->_id,
                'employer_id'          => $jobOffer->employer_id,
                'user_id'              => $validated['is_logged_in'] ? auth()->id() : null,
                'type_id'              => $applicationType->id,
                'candidate_first_name' => $validated['first_name'] ?? null,
                'candidate_last_name'  => $validated['last_name'] ?? null,
                'candidate_email'      => $validated['email'] ?? null,
                'candidate_phone'      => $validated['phone'] ?? null,
                'cv_file'              => $validated['cv_file'] ?? null,
            ]);

            $filesData = [];
            if ($request->hasFile('cv_file')) {
                $file        = $request->file('cv_file');
                $path        = $file->store("applications/{$application->_id}", 'public');
                $filesData = [
                    'path'          => $path,
                    'original_name' => $file->getClientOriginalName(),
                    'size'          => $file->getSize(),
                    'mime_type'     => $file->getMimeType(),
                    'uploaded_at'   => now(),
                ];
                $application->update(['cv_file' => $filesData]);
            }
        } catch (\Exception $e) {
            Storage::disk('public')->deleteDirectory("applications/{$application->_id}");
            $application->delete();

            return $this->error('Nie można złożyć aplikacji. Proszę spróbować ponownie później.', 500);
        }

        return $this->success(new ApplicationResource($application), [], 'Aplikacja została pomyślnie złożona.');
    }

    public function update(Request $request, string $id): JsonResponse
    {
        return $this->notImplemented('update');
    }

    public function destroy(string $id): JsonResponse
    {
        return $this->notImplemented('destroy');
    }
}
