<?php

namespace MbeGroup\Job\Jobs;

use App\ValueObject\Status;
use MbeGroup\Job\Models\Import;
use MbeGroup\Job\Models\JobOffer;
use MbeGroup\Job\Models\ArchivedJobOffer;
use MbeGroup\Job\Providers\ImportProviders\ProviderFactory;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Throwable;
use MbeGroup\Job\Contracts\JobOfferArchiveServiceInterface;

class ProcessImportJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $timeout = 600;
    public int $tries   = 1;

    public function __construct(public string $importId, protected JobOfferArchiveServiceInterface $archiveService)
    {
    }

    public function handle(ProviderFactory $factory): void
    {
        $import = Import::find($this->importId);

        if (!$import) {
            throw new \RuntimeException("Import with ID {$this->importId} not found");
        }

        $import->update(['is_running' => true]);

        $runId     = (string) Str::uuid();
        $startTime = now();

        try {
            DB::beginTransaction();

            $provider     = $factory->make($import);
            $rawOffers    = $provider->fetchOffers();
            $parsedOffers = $provider->parseOffers($rawOffers['offers']);
            $stats        = $this->reconcile($import, $parsedOffers);

            $this->validateCounts($parsedOffers, $stats);

            DB::commit();

            $import->addRunToHistory([
                'run_id'                            => $runId,
                'status'                            => 'success',
                'started_at'                        => $startTime->toISOString(),
                'completed_at'                      => now()->toISOString(),
                'duration_seconds'                  => $startTime->diffInSeconds(now()),
                'finded_offers_count'               => $rawOffers['finded_offers_count'],
                'offers_count_after_location_split' => $rawOffers['offers_count_after_location_split'],
                'fetched_count'                     => count($parsedOffers),
                'new_count'                         => $stats['new'],
                'restored_count'                    => $stats['restored'],
                'active_count'                      => $stats['active'],
                'archived_count'                    => $stats['archived'],
                'new_offers'                        => $stats['new_offers'],
                'restored_offers'                   => $stats['restored_offers'],
                'active_offers'                     => $stats['active_offers'],
                'archived_offers'                   => $stats['archived_offers'],
                'existing_offers'                   => $stats['existing_offers'],
            ]);

            $import->update([
                'last_run_at' => now(),
                'is_running'  => false,
            ]);

        } catch (Throwable $e) {
            DB::rollBack();

            $import->addRunToHistory([
                'run_id'           => $runId,
                'status'           => 'failed',
                'started_at'       => $startTime->toISOString(),
                'completed_at'     => now()->toISOString(),
                'duration_seconds' => $startTime->diffInSeconds(now()),
                'error_message'    => $e->getMessage(),
                'error_file'       => $e->getFile(),
                'error_line'       => $e->getLine(),
            ]);

            $import->update(['is_running' => false]);

            throw $e;
        }
    }

      /**
     * Reconcile imported offers with database
     * Returns statistics about the reconciliation
     */
    protected function reconcile(Import $import, array $offers): array
    {
        $providerIds = collect($offers)->pluck('provider_id')->unique()->values()->toArray();

        $stats = [
            'new'             => 0,
            'restored'        => 0,
            'active'          => 0,
            'archived'        => 0,
            'existing'        => 0,
            'new_offers'      => [],
            'restored_offers' => [],
            'active_offers'   => [],
            'archived_offers' => [],
            'existing_offers' => [],
        ];

        foreach ($offers as $offerData) {
            $providerId = $offerData['provider_id'];

            $existing = JobOffer::where('provider_id', $providerId)->first();

            if ($existing) {
                if ($existing->status->isActive()) {
                    $stats['active']++;
                    $stats['active_offers'][0][] = [
                        'id'    => $existing->id,
                        'title' => $existing->title,
                    ];
                } else {
                    $stats['existing']++;
                    $stats['existing_offers'][0][] = [
                        'id'    => $existing->id,
                        'title' => $existing->title,
                    ];
                }
                continue;
            }

            $archived = ArchivedJobOffer::where('provider_id', $providerId)->first();

            if ($archived) {
                $restoredOffer = $this->archiveService->restoreOffer($archived);
                $stats['restored']++;
                $stats['restored_offers'][0][] = [
                    'id'    => $restoredOffer->id,
                    'title' => $restoredOffer->title,
                ];
            } else {
                $newOffer = JobOffer::create($offerData);
                $stats['new']++;
                $stats['new_offers'][0][] = [
                    'id'    => $newOffer->id,
                    'title' => $newOffer->title,
                ];
            }
        }

        $toArchive = JobOffer::where('employer_id', $import->employer_id)
            ->whereNotIn('provider_id', $providerIds)
            ->get();

        foreach ($toArchive as $job) {
            $this->archiveService->archiveOffer($job, 'not_in_import');
            $stats['archived']++;
            $stats['archived_offers'][0][] = [
                'id'    => $job->id,
                'title' => $job->title,
            ];
        }

        return $stats;
    }

      /**
     * Validate that counts match expected values
     */
    protected function validateCounts(array $parsedOffers, array $stats): void
    {
        $expectedCount = count($parsedOffers);
        $actualCount   = $stats['new'] + $stats['restored'] + $stats['existing'] + $stats['active'];

        if ($expectedCount !== $actualCount) {
            throw new \RuntimeException(
                "Import synchronization failed - count mismatch. " .
                "Expected: {$expectedCount}, Actual: {$actualCount} " .
                "(new: {$stats['new']}, restored: {$stats['restored']}, active: {$stats['active']}, existing: {$stats['existing']}). " .
                "Import has been rolled back. Please check the data and try again."
            );
        }
    }
}
