<?php

namespace MbeGroup\Job\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use MbeGroup\Job\Models\JobOffer;
use Elastic\Elasticsearch\Client;

class JobOffersReindex extends Command
{
    protected $signature = 'job-offers:reindex
                            {--flush : Usuń wszystkie dokumenty przed reindeksacją}
                            {--no-queue : Indeksuj synchronicznie (bez kolejki)}';

    protected $description = 'Reindeksuj oferty pracy w Elasticsearch';

    public function handle()
    {
        $this->info('🚀 Start reindeksacji ofert pracy...');
        $this->newLine();

        // Krok 1: Graceful restart queue workera
        $this->info('🔄 Graceful restart queue workera...');
        Artisan::call('queue:restart');
        $this->info('✅ Queue worker zrestartuje się po zakończeniu obecnych jobów');
        $this->warn('⏳ Czekam 5 sekund...');
        sleep(5);
        $this->newLine();

        // Krok 2: Wyczyść cache
        $this->info('⏳ Czyszczenie cache...');
        Artisan::call('optimize:clear');
        $this->info('✅ Cache wyczyszczony');
        $this->newLine();

        // Krok 3: Sprawdź liczbę ofert
        $count = JobOffer::count();
        $this->info("📊 Ofert w MongoDB: {$count}");
        $this->newLine();

        if ($count === 0) {
            $this->warn('⚠️  Brak ofert do reindeksacji!');
            return Command::SUCCESS;
        }

        // Krok 4: Opcjonalnie usuń istniejące dokumenty
        if ($this->option('flush')) {
            $this->warn('🗑️  Usuwanie istniejących dokumentów z Elasticsearch...');

            if ($this->confirm('Czy na pewno chcesz usunąć wszystkie dokumenty? (spowoduje downtime)', false)) {
                $this->call('scout:flush', ['searchable' => 'MbeGroup\\Job\\Models\\JobOffer']);
                $this->info('✅ Dokumenty usunięte');
            } else {
                $this->warn('❌ Anulowano usuwanie');
                return Command::FAILURE;
            }

            $this->newLine();
        }

        // Krok 5: Ustaw tryb indeksacji
        if ($this->option('no-queue')) {
            $this->info('⚡ Indeksacja synchroniczna (bez kolejki)...');
            config(['scout.queue' => false]);
        } else {
            $this->info('🔄 Indeksacja asynchroniczna (przez kolejkę)...');
        }

        // Krok 6: Import
        $this->info('📥 Rozpoczynam import...');
        $this->newLine();

        try {
            $this->call('scout:import', ['searchable' => 'MbeGroup\\Job\\Models\\JobOffer']);
        } catch (\Exception $e) {
            // Ignoruj błąd "Bulk update error" - to false positive
            if (!str_contains($e->getMessage(), 'Bulk update error')) {
                throw $e;
            }
            $this->warn('⚠️  Otrzymano błąd importu (może być false positive)');
        }

        $this->newLine();

        // Krok 7: AUTOMATYCZNE CZYSZCZENIE I NAPRAWA ALIASÓW
        $this->info('🧹 Czyszczenie starych indeksów i naprawianie aliasów...');
        $this->cleanupIndicesAndAliases();
        $this->newLine();

        // Krok 8: Przetwórz joby jeśli przez kolejkę
        if (!$this->option('no-queue')) {
            $this->info('⏳ Joby zostały dodane do kolejki Scout.');
            $this->info('💡 Worker queue-scout przetworzy je w tle (~1-2 min)');
            $this->info('💡 Sprawdź status: docker compose logs queue-scout -f');
            $this->newLine();

            $this->warn('Czekam 10 sekund na przetworzenie jobów...');
            sleep(10);

            // Napraw aliasy po krótkim oczekiwaniu
            $this->info('🧹 Finalne czyszczenie...');
            $this->cleanupIndicesAndAliases();
        }

        $this->newLine();

        // Krok 9: Sprawdź wynik
        $client = app(Client::class);
        try {
            $count = $client->count(['index' => 'job_offers'])->asArray();
            $documentCount = $count['count'] ?? 0;

            if ($documentCount > 0) {
                $this->info("🎉 Reindeksacja zakończona pomyślnie! Dokumentów w ES: {$documentCount}");
            } else {
                $this->warn("⚠️  Reindeksacja zakończona, ale brak dokumentów w ES (może być problem z aliasem)");
                $this->info("💡 Sprawdź ręcznie: curl 'http://localhost:9200/job_offers/_count?pretty'");
            }
        } catch (\Exception $e) {
            $this->warn("⚠️  Nie można sprawdzić liczby dokumentów: " . $e->getMessage());
        }

        return Command::SUCCESS;
    }

    /**
     * Czyści stare indeksy i naprawia aliasy (usuwa routing)
     */
    private function cleanupIndicesAndAliases(): void
    {
        try {
            $client = app(Client::class);

            // Pobierz wszystkie indeksy job_offers_*
            $indices = $client->cat()->indices(['index' => 'job_offers_*', 'format' => 'json'])->asArray();
            $allIndices = collect($indices)->pluck('index')->toArray();

            if (empty($allIndices)) {
                $this->warn('  ⚠️  Brak indeksów job_offers_*');
                return;
            }

            $this->line("  Znaleziono indeksów: " . count($allIndices));

            if (count($allIndices) === 1) {
                $singleIndex = $allIndices[0];
                $this->line("  Pojedynczy indeks: {$singleIndex}");

                // Napraw alias (usuń routing)
                $this->fixAlias($client, $singleIndex);

            } else {
                // Wiele indeksów - usuń stare, zostaw najnowszy
                rsort($allIndices); // Sortuj malejąco (najnowszy pierwszy)
                $newestIndex = array_shift($allIndices);

                $this->line("  Najnowszy indeks: {$newestIndex}");

                // Usuń stare indeksy
                foreach ($allIndices as $oldIndex) {
                    try {
                        $client->indices()->delete(['index' => $oldIndex]);
                        $this->line("  ✅ Usunięto stary indeks: {$oldIndex}");
                    } catch (\Exception $e) {
                        $this->warn("  ⚠️  Nie można usunąć {$oldIndex}: " . $e->getMessage());
                    }
                }

                // Napraw alias dla najnowszego
                $this->fixAlias($client, $newestIndex);
            }

        } catch (\Exception $e) {
            $this->error("  ❌ Błąd podczas czyszczenia: " . $e->getMessage());
        }
    }

    /**
     * Naprawia alias (usuwa routing)
     */
    private function fixAlias(Client $client, string $indexName): void
    {
        try {
            // Usuń alias jeśli istnieje
            try {
                $client->indices()->deleteAlias([
                    'index' => $indexName,
                    'name' => 'job_offers'
                ]);
            } catch (\Exception $e) {
                // Alias może nie istnieć - OK
            }

            // Dodaj alias bez routingu
            $client->indices()->putAlias([
                'index' => $indexName,
                'name' => 'job_offers'
            ]);

            $this->line("  ✅ Naprawiono alias: job_offers → {$indexName}");

        } catch (\Exception $e) {
            $this->warn("  ⚠️  Nie można naprawić aliasu: " . $e->getMessage());
        }
    }
}
