<?php

namespace MbeGroup\Job\Requests;

use Illuminate\Foundation\Http\FormRequest;
use MbeGroup\Job\Models\JobOffer;
use App\Models\Dictionary;

class StoreApplicationRequest extends FormRequest
{
    protected ?JobOffer $jobOffer          = null;
    protected ?Dictionary $applicationType = null;

    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules()
    {
        if ($this->has('type_id')) {
            $this->applicationType = Dictionary::find($this->input('type_id'));
        }

        $requiresData = $this->applicationType && $this->applicationType->slug === 'form-application';

        return [
            'job_offer_id' => [
                'required',
                'string',
                function ($attribute, $value, $fail) {
                    $this->jobOffer = JobOffer::find($value);
                    if (!$this->jobOffer) {
                        $fail('Wybrana oferta pracy nie istnieje.');
                    }
                },
            ],
             'type_id' => [
                'required',
                'exists:dictionaries,id'
            ],
            'first_name'   => [$requiresData ? 'required' : 'nullable', 'string', 'max:255'],
            'last_name'    => [$requiresData ? 'required' : 'nullable', 'string', 'max:255'],
            'email'        => [$requiresData ? 'required' : 'nullable', 'email'],
            'phone'        => ['nullable', 'string'],
            'is_logged_in' => ['required', 'boolean'],
            'cv_file'        => ['nullable'],
            'cv_file.*'      => ['file', 'max:10240', 'mimes:pdf,doc,docx,txt,odt'],
        ];
    }

    public function messages()
    {
        return [
            'job_offer_id.required'       => 'Pole oferta pracy jest wymagane.',
            'job_offer_id.string'         => 'Pole oferta pracy musi być ciągiem znaków.',
            'type_id.required'            => 'Pole typ aplikacji jest wymagane.',
            'type_id.exists'              => 'Wybrany typ aplikacji jest nieprawidłowy.',
            'first_name.required_without' => 'Pole imię jest wymagane, jeśli użytkownik nie jest zalogowany.',
            'first_name.string'           => 'Pole imię musi być ciągiem znaków.',
            'first_name.max'              => 'Pole imię nie może być dłuższe niż 255 znaków.',
            'last_name.required_without'  => 'Pole nazwisko jest wymagane, jeśli użytkownik nie jest zalogowany.',
            'last_name.string'            => 'Pole nazwisko musi być ciągiem znaków.',
            'last_name.max'               => 'Pole nazwisko nie może być dłuższe niż 255 znaków.',
            'email.required_without'      => 'Pole email jest wymagane, jeśli użytkownik nie jest zalogowany.',
            'email.email'                 => 'Podaj prawidłowy adres email.',
            'cv_file.required_if'         => 'Plik CV jest wymagany, jeśli użytkownik jest zalogowany.',
            'cv_file.file'                => 'Przesłany plik CV musi być prawidłowym plikiem.',
            'cv_file.max'                 => 'Plik CV nie może być większy niż 10MB.',
            'cv_file.mimes'               => 'Plik CV musi być w formacie PDF, DOC lub DOCX.',
        ];
    }

    public function getJobOffer(): JobOffer
    {
        return $this->jobOffer;
    }

    public function getApplicationType(): Dictionary
    {
        return $this->applicationType;
    }
}
