<?php

namespace MbeGroup\Job\Services;

use Carbon\Carbon;
use MbeGroup\Job\Contracts\JobOfferRecommendationServiceInterface;
use MbeGroup\Job\Models\Application;
use MbeGroup\Job\Models\JobOffer;
use MbeGroup\Job\Models\Recommendation;
use MbeGroup\Job\Resources\RecommendationJobOfferResource;

class JobOfferRecommendationService implements JobOfferRecommendationServiceInterface
{
    private const MAX_RECOMMENDATIONS = 10;

    const WEIGHT = [
        'apply'        => 5,
        'favorites'    => 4,
        'user_profile' => 3,
        'viewed'       => 2,
        'search'       => 1
    ];

    public function findUserJobRecomendationByType(int $userId, string $type, array $excludedFilters = []): array
    {
        $recommendation = Recommendation::where('user_id', $userId)->first()->toArray();
        if (!$recommendation || !isset($recommendation['filter_data'][$type]) || !is_array($recommendation['filter_data'][$type])) {
            return [];
        }
        $newRecommendation['filter_data'][$type] = $recommendation['filter_data'][$type];
        if (!empty($excludedFilters)) {
            $lastIndex = count($recommendation['filter_data'][$type]) - 1;

            $recommendation['filter_data']['excluded_filters'][$lastIndex]['ids']       = $excludedFilters;
            $recommendation['filter_data']['excluded_filters'][$lastIndex]['timestamp'] = (new \DateTimeImmutable())->getTimestamp();
        }
        $newRecommendation['filter_data']['excluded_filters'] = $recommendation['filter_data']['excluded_filters'] ?? [];
        $excludedIds                                          = $this->getExludeIds($newRecommendation['filter_data']['excluded_filters'] ?? [], $userId);

        return $this->getJobOfferList($newRecommendation, $excludedIds);
    }

    public function prepareDataFromRequest(array $requestData): array
    {
        $data['user_id']     = auth()->user() ? auth()->user()->id : null;
        $typeFilterData      = $requestData['filter_data'] ?? [];
        $data['filter_data'] = [$requestData['type'] => $typeFilterData];

        return $data;
    }

    public function addFilterDataToUserRecommendation(array $data, string $type): array
    {
        $existingRecommendation = Recommendation::where('user_id', $data['user_id'])->first();
        if (!$existingRecommendation) {
            return $this->createNewRecommendation($data, $type);
        }

        return $this->updateExistingRecommendation($existingRecommendation->toArray(), $data, $type);
    }

    public function findUserJobRecomendation(int $userId): array
    {
        $recommendation = Recommendation::where('user_id', $userId)->first()->toArray();
        if (!$recommendation || !isset($recommendation['filter_data']) || !is_array($recommendation['filter_data'])) {
            return [];
        }
        $excludedIds = $this->getExludeIds($recommendation['filter_data']['excluded_filters'] ?? [], $userId);

        return $this->getJobOfferList($recommendation, $excludedIds);
    }

    /**
     * Get job offer list based on filter data and exclude IDs
     *
     * @param array $filterData
     * @param array $excludedIds
     * @param string|null $routeName
     * @param bool|null $isNew
     *
     * @return array
     */
    private function getJobOfferList(array $filterData = [], array $excludedIds = [], $routeName = null, $isNew = null): array
    {
        $jobOfferSearchService = app()->make(JobOffersSearchService::class);
        $result                = [];

        foreach ($filterData['filter_data'] as $type => $data) {
            if ($type === 'excluded_filters') {
                continue;
            }

            if (isset($data['timestamp'])) {
                usort($data, function($a, $b) {
                   if (is_array($a) && is_array($b)) {
                    $timestampA = $a['timestamp'] instanceof Carbon
                        ? $a['timestamp']->toDateTime()->getTimestamp()
                        :         0;
                    $timestampB = $b['timestamp'] instanceof Carbon
                        ? $b['timestamp']->toDateTime()->getTimestamp()
                        :         0;
                    return $timestampB - $timestampA;
                    }
                });
            }

            if (empty($data)) {
                $mergedExcludeIds = array_unique(array_merge($excludedIds, $value['excludedIds'] ?? []));
                $searchResult     = $jobOfferSearchService->search(
                    [
                        'exclude_ids'           => $mergedExcludeIds,
                        'publication_date_from' => $isNew ? date('Y-m-d', strtotime('-7 days')) : null,
                    ],
                    10,
                    1,
                );

                $jobOfferList = RecommendationJobOfferResource::collection($searchResult)->resolve();
                if (empty($jobOfferList)) {
                    continue;
                }

                $result[$type][] = [
                    'job_offers' => $jobOfferList,
                    'timestamp'  => $value['timestamp'] ?? null,
                ];
            } else {

                foreach ($data as $key => $value) {
                    if (!is_array($value) || empty($value)) {
                        continue;
                    }
                    $mergedExcludeIds = array_unique(array_merge($excludedIds, $value['excludedIds'] ?? []));

                    $searchResult     = $jobOfferSearchService->search(
                        [
                            'q'                          => $value['keyword'] ?? '',
                            'employer_ids'               => !empty($value['employer_ids']) ? implode(',', $value['employer_ids']) : '',
                            'industry_ids'               => !empty($value['industry_ids']) ? implode(',', $value['industry_ids']) : '',
                            'specialization_ids'         => !empty($value['specialization_ids']) ? implode(',', $value['specialization_ids']) : '',
                            'position_ids'               => !empty($value['position_ids']) ? implode(',', $value['position_ids']) : '',
                            'position_level_ids'         => !empty($value['position_level_ids']) ? implode(',', $value['position_level_ids']) : '',
                            'operating_mode_ids'         => !empty($value['operating_mode_ids']) ? implode(',', $value['operating_mode_ids']) : '',
                            'salary_employment_type_ids' => !empty($value['salary_employment_type_ids']) ? implode(',', $value['salary_employment_type_ids']) : '',
                            'hourly_rate_ids'            => !empty($value['hourly_rate_ids']) ? implode(',', $value['hourly_rate_ids']) : '',
                            'only_with_salary'           => $value['only_with_salary'] ?? false,
                            'min_salary'                 => $value['min_salary'] ?? null,
                            'preferred_study_ids'        => !empty($value['preferred_study_ids']) ? implode(',', $value['preferred_study_ids']) : '',
                            'workplace_type_ids'         => !empty($value['workplace_type_ids']) ? implode(',', $value['workplace_type_ids']) : '',
                            'work_system_id'             => $value['work_system_id'] ?? null,
                            'delegations_id'             => $value['delegations_id'] ?? null,
                            'contract_duration_id'       => $value['contract_duration_id'] ?? null,
                            'salary_structure_ids'       => !empty($value['salary_structure_ids']) ? implode(',', $value['salary_structure_ids']) : '',
                            'has_team_management'        => $value['has_team_management'] ?? null,
                            'has_driving_license'        => $value['has_driving_license'] ?? null,
                            'no_experience'              => $value['no_experience'] ?? null,
                            'has_company_car'            => $value['has_company_car'] ?? null,
                            'is_remote_recruitment'      => $value['is_remote_recruitment'] ?? null,
                            'is_for_technicians'         => $value['is_for_technicians'] ?? null,
                            'company_size_ids'           => !empty($value['company_size_ids']) ? implode(',', $value['company_size_ids']) : '',
                            'company_type_ids'           => !empty($value['company_type_ids']) ? implode(',', $value['company_type_ids']) : '',
                            'lat'                        => $value['lat'] ?? null,
                            'lon'                        => $value['lon'] ?? null,
                            'distance'                   => $value['distance'] ?? 50,
                            'exclude_ids'                => $mergedExcludeIds,
                            'publication_date_from'      => $isNew ? date('Y-m-d', strtotime('-7 days')) : null,
                        ],
                        10,
                        1,
                    );

                    $jobOfferList = RecommendationJobOfferResource::collection($searchResult)->resolve();
                    if (empty($jobOfferList)) {
                        continue;
                    }

                    $result[$type][$key] = [
                        'job_offers' => $jobOfferList,
                        'timestamp'  => $value['timestamp'] ?? null,
                    ];
                }
            }
        }

        return $this->sortRecommendationsByWeight($result);
    }

        /**
     * Sort recommendations by weight and timestamp
     *
     * @param array $recommendations
     *
     * @return array
     */
    private function sortRecommendationsByWeight(array $recommendations): array
    {
        $result = [];
        foreach ($recommendations as $type => $value) {
            $jobOffersList = [];
            usort($value, function($a, $b) {
            $timestampA = isset($a['timestamp']) && $a['timestamp'] instanceof Carbon
                ? $a['timestamp']->toDateTime()->getTimestamp()
                :         0;
            $timestampB = isset($b['timestamp']) && $b['timestamp'] instanceof Carbon
                ? $b['timestamp']->toDateTime()->getTimestamp()
                :         0;
            return $timestampB - $timestampA;
            });

            foreach ($value as $filterData) {
                if (!isset($filterData['job_offers']) || empty($filterData['job_offers'])) {
                    continue;
                }
                array_push($jobOffersList, ...$filterData['job_offers']);
            }

            $result[$type] = $jobOffersList;
        }

        $sortedResult = [];
        $usedJobIds   = [];

        foreach (self::WEIGHT as $type => $weight) {
            if (isset($result[$type])) {
                $uniqueJobOffers = [];

                foreach ($result[$type] as $jobOffer) {
                    $jobId = $jobOffer['id'] ?? null;
                    if ($jobId === null || in_array($jobId, $usedJobIds)) {
                        continue;
                    }
                    $uniqueJobOffers[] = $jobOffer;
                    $usedJobIds[]      = $jobId;
                }

                $sortedResult[$type] = $uniqueJobOffers;
            }
        }

        $finalJobOffers = [];
        foreach ($sortedResult as $type => $jobOffers) {
            $finalJobOffers = array_merge($finalJobOffers, $jobOffers);
        }

        return $finalJobOffers;
    }

    /**
     * Get excluded job offer IDs based on user applications and filters
     *
     * @param array $excludedFilters
     * @param int|null $userId
     *
     * @return array
     */
    private function getExludeIds(array $excludedFilters, ?int $userId): array
    {
        if (!$userId) {
            $excludedIdsByFilter = [];
            foreach ($excludedFilters as $item) {
                if (isset($item['ids']) && is_array($item['ids'])) {
                    foreach ($item['ids'] as $id) {
                        $excludedIdsByFilter[] = $id;
                    }
                }
            }
            return array_values(array_unique($excludedIdsByFilter));
        }
        $userApplications       = Application::where('user_id', $userId)->get();
        $applicationJobOfferIds = $userApplications->pluck('job_offer_id')->toArray();
        $jobOffersConnectionIds = JobOffer::whereIn('_id', $applicationJobOfferIds)->pluck('connection_id')->toArray();

        $excludedIdsByFilter = [];
        foreach ($excludedFilters as $item) {
            if (isset($item['ids']) && is_array($item['ids'])) {
                foreach ($item['ids'] as $id) {
                    $excludedIdsByFilter[] = $id;
                }
            }
        }

        return array_values(array_unique(array_merge(
            $excludedIdsByFilter,
            $jobOffersConnectionIds
        )));
    }

    /**
     * Utwórz nową rekomendację
     *
     * @param array $recommendation
     * @param string $type
     *
     * @return array
     */
    private function createNewRecommendation(array $recommendation, string $type): array
    {
        if (isset($recommendation['filter_data']) && !empty($type)) {
            $recommendation['filter_data'] = $this->restructureFilterData($recommendation['filter_data']);
        }

        $this->addTimestampToFilterType($recommendation, $type);

        $newRecommendation = Recommendation::create($recommendation);

        return $newRecommendation ? $newRecommendation->toArray(): [];
    }

    /**
     * Zaktualizuj istniejącą rekomendację
     *
     * @param array $existing
     * @param array $new
     * @param string $type
     *
     * @return array
     */
    private function updateExistingRecommendation(array $existing, array $new, string $type): array
    {
        $existingRecommendations        = $existing['filter_data'][$type] ?? [];
        $newRecommendation              = $this->extractNewRecommendation($new, $type);
        $newRecommendation['timestamp'] = (new \DateTimeImmutable())->getTimestamp();
        $updatedRecommendations         = $this->manageRecommendationsLimit($existingRecommendations, $newRecommendation);
        if ($type === 'user_profile') {
            $recommendation = Recommendation::where('id', $existing['id'])
                ->update([
                    'filter_data.' . $type . '.0' => $newRecommendation
                ]);
        } else {
            $recommendation = Recommendation::where('id', $existing['id'])
                ->update([
                    'filter_data.' . $type => $updatedRecommendations
                ]);
        }

        return $recommendation ? Recommendation::find($existing['id'])->toArray(): [];
    }

    /**
     * Zarządzaj limitem rekomendacji (max 10)
     *
     * @param array $existingRecommendations
     * @param array $newRecommendation
     *
     * @return array
     */
    private function manageRecommendationsLimit(array $existingRecommendations, array $newRecommendation): array
    {
        if (count($existingRecommendations) < self::MAX_RECOMMENDATIONS) {
            $existingRecommendations[] = $newRecommendation;
            return $existingRecommendations;
        }

        $oldestIndex                           = $this->findOldestRecommendationIndex($existingRecommendations);
        $existingRecommendations[$oldestIndex] = $newRecommendation;

        return $existingRecommendations;
    }

    /**
     * Znajdź indeks najstarszej rekomendacji
     *
     * @param array $recommendations
     *
     * @return int
     */
    private function findOldestRecommendationIndex(array $recommendations): int
    {
        $oldestIndex     = 0;
        $oldestTimestamp = null;

        foreach ($recommendations as $index => $recommendation) {
            $currentTimestamp = $recommendation['timestamp'] ?? (new \DateTimeImmutable('1970-01-01'))->getTimestamp();

            if ($oldestTimestamp === null || $currentTimestamp < $oldestTimestamp) {
                $oldestTimestamp = $currentTimestamp;
                $oldestIndex     = $index;
            }
        }

        return $oldestIndex;
    }

    /**
     * Wyciągnij nową rekomendację z danych wejściowych
     *
     * @param array $recommendation
     * @param string $type
     *
     * @return array
     */
    private function extractNewRecommendation(array $recommendation, string $type): array
    {
        return $recommendation['filter_data'][$type][0] ?? $recommendation['filter_data'][$type];
    }

    /**
     * Restrukturyzuj filter_data
     *
     * @param array $filterData
     *
     * @return array
     */
    private function restructureFilterData(array $filterData): array
    {
        $restructured = [];

        foreach ($filterData as $key => $value) {
            if (is_array($value)) {
                $restructured[$key] = [];
                foreach ($value as $subkey => $subvalue) {
                    if (!isset($restructured[$key][0])) {
                        $restructured[$key][0] = [];
                    }
                    $restructured[$key][0][$subkey] = $subvalue;
                }
            } else {
                $restructured[$key] = $value;
            }
        }

        return $restructured;
    }

    /**
     * Dodaj timestamp do filter_data dla konkretnego typu
     *
     * @param array $recommendation
     * @param string $type
     *
     * @return void
     */
    private function addTimestampToFilterType(array &$recommendation, string $type): void
    {
        if (!isset($recommendation['filter_data'][$type])) {
            return;
        }

        $timestamp = (new \DateTimeImmutable())->getTimestamp();

        if (isset($recommendation['filter_data'][$type][0])) {
            $recommendation['filter_data'][$type][0]['timestamp'] = $timestamp;
        } else {
            $recommendation['filter_data'][$type]['timestamp'] = $timestamp;
        }
    }
}
