<?php

namespace MbeGroup\Job\Providers\ImportProviders;

use App\ValueObject\Status;
use MbeGroup\Job\Models\Import;
use Illuminate\Support\Str;
use MbeGroup\Job\Contracts\ScraperJobOfferServiceInterface;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;

abstract class AbstractProvider
{
    protected Import $import;
    protected ScraperJobOfferServiceInterface $scraperService;
    protected string $sessionToken;

    public function __construct(Import $import, ScraperJobOfferServiceInterface $scraperService)
    {
        $this->import         = $import;
        $this->scraperService = $scraperService;
        $this->sessionToken   = Str::uuid()->toString();
    }

          /**
     * Fetch raw offers from external API
     * Must return array of raw offer data
     */
    abstract public function fetchOffers(): array;

      /**
     * Transform single raw offer to JobOffer format
     * Can return single offer or array of offers (location explosion)
     */
    abstract protected function transformOffer(array $rawOffer): array;

      /**
     * Parse all fetched offers
     * Handles explosion and adds employer/template data
     */
    public function parseOffers(array $rawOffers): array
    {
        $parsed = [];

        foreach ($rawOffers as $rawOffer) {
            $transformed = $this->transformOffer($rawOffer);

              // Check if explosion happened (array of offers vs single offer)
            $offers = $this->isMultipleOffers($transformed) ? $transformed : [$transformed];

            foreach ($offers as $offer) {
                $parsed[] = $this->enrichOffer($offer);
            }
        }

        return $parsed;
    }

      /**
     * Enrich offer with import-level data
     */
    protected function enrichOffer(array $offer): array
    {
        return array_merge($offer, [
            'employer_id'             => $this->import->employer_id,
            'job_offer_template_name' => $this->import->template_name,
        ]);
    }

      /**
     * Check if transformed result is multiple offers
     */
    protected function isMultipleOffers(array $data): bool
    {
          // If first element is array with provider_id, it's multiple offers
        return isset($data[0]) && is_array($data[0]) && isset($data[0]['provider_id']);
    }

      /**
     * Generate unique provider ID
     */
    protected function makeProviderId(string $referenceNumber, string $location): string
    {
        return sprintf(
            '%s_%s_%s',
            $this->import->provider_type,
            Str::slug($referenceNumber),
            Str::slug($location)
        );
    }

      /**
     * Get config value from import
     */
    protected function config(string $key, mixed $default = null): mixed
    {
        return data_get($this->import->config, $key, $default);
    }

      /**
     * Validate required config keys
     */
    protected function validateConfig(array $requiredKeys): void
    {
        $missing = [];

        foreach ($requiredKeys as $key) {
            if (!isset($this->import->config[$key])) {
                $missing[] = $key;
            }
        }

        if (!empty($missing)) {
            throw new \InvalidArgumentException(
                "Missing required config keys: " . implode(', ', $missing)
            );
        }
    }

    protected function searchPlaceByName(string $placeName): ?array
    {
        $placeName = trim($placeName);

        $cacheKey = 'osm_location_' . md5(strtolower($placeName));

        return Cache::remember($cacheKey, now()->addDays(90), function () use ($placeName, $cacheKey) {

            $params = [
                'format' => 'json',
                'addressdetails' => 1,
                'limit' => 1,
            ];

            if (preg_match('/^([^,]+),\s*(?:ul\.\s*)?(.+?)\s+(\d+[a-zA-Z]?)$/i', $placeName, $matches)) {
                $params['city'] = trim($matches[1]);
                $params['street'] = trim($matches[2]) . ' ' . trim($matches[3]);
            }
            elseif (preg_match('/^woj\.\s*(.+)$/i', $placeName, $matches)) {
                $params['state'] = trim($matches[1]);
            }
            else {
                $params['city'] = $placeName;
            }

            sleep(1);

            $response = Http::withHeaders([
                'User-Agent' => config('app.name', 'FilamentApp'),
            ])->get(config('app.open_street_map.endpoint'), $params);

            $data = $response->json();

            if (empty($data)) {
                $response = Http::withHeaders([
                    'User-Agent' => config('app.name', 'FilamentApp'),
                ])->get(config('app.open_street_map.endpoint'), [
                    'q' => $placeName,
                    'format' => 'json',
                    'addressdetails' => 1,
                    'limit' => 1,
                ]);

                $data = $response->json();
            }

            if (empty($data)) {
                return [
                    'location_name'       => $placeName,
                    'municipality'        => null,
                    'region'              => null,
                    'postal_code'         => null,
                    'street'              => null,
                    'building_number'     => null,
                    'latitude'            => null,
                    'longitude'           => null,
                    'phone'               => null,
                    'email'               => null,
                    'website'             => null,
                    'status'              => Status::ACTIVE,
                    'location_is_default' => true,
                ];
            }

            $address = $data[0]['address'] ?? [];

            return [
                'location_name'       => $address['city'] ?? $address['town'] ?? $address['village'] ?? $placeName,
                'municipality'        => $address['county'] ?? null,
                'region'              => $address['state'] ?? null,
                'postal_code'         => $address['postcode'] ?? null,
                'street'              => $address['road'] ?? null,
                'building_number'     => $address['house_number'] ?? null,
                'latitude'            => $data[0]['lat'] ?? null,
                'longitude'           => $data[0]['lon'] ?? null,
                'phone'               => null,
                'email'               => null,
                'website'             => null,
                'status'              => Status::ACTIVE,
                'location_is_default' => true,
            ];
        });
    }
}
