<?php

namespace MbeGroup\Job\Services;

use MbeGroup\Job\Models\JobOffer;
use MbeGroup\Job\Models\ArchivedJobOffer;
use MbeGroup\Job\Contracts\JobOfferArchiveServiceInterface;
use App\ValueObject\Status;
use Illuminate\Support\Facades\DB;

class JobOfferArchiveService implements JobOfferArchiveServiceInterface
{
    public function archiveOffer(JobOffer $jobOffer, ?string $reason = null): ArchivedJobOffer
    {
        return DB::transaction(function () use ($jobOffer, $reason) {
            $attributes = $jobOffer->getAttributes();

            if ($jobOffer->shouldBeSearchable()) {
                $jobOffer->unsearchable();
            }

            unset($attributes['product_statistics']);

            $attributes['archived_at'] = now();
            $attributes['archived_reason'] = $reason;

            $archivedOffer = new ArchivedJobOffer();
            $archivedOffer->setRawAttributes($attributes);
            $archivedOffer->status = Status::INACTIVE;
            $archivedOffer->save();

            $jobOffer->delete();

            return $archivedOffer;
        });
    }

    public function archiveOffers(array $offerIds, ?string $reason = null): array
    {
        return DB::transaction(function () use ($offerIds, $reason) {
            $archived = [];

            $jobOffers = JobOffer::whereIn('_id', $offerIds)->get();

            foreach ($jobOffers as $jobOffer) {
                $archived[] = $this->archiveOffer($jobOffer, $reason);
            }

            return $archived;
        });
    }

    public function restoreOffer(ArchivedJobOffer $archivedOffer): JobOffer
    {
        return DB::transaction(function () use ($archivedOffer) {
            $attributes = $archivedOffer->getAttributes();

            unset($attributes['archived_at'], $attributes['archived_reason']);
            $attributes['status'] = Status::DRAFT;

            $jobOffer = new JobOffer();
            $jobOffer->setRawAttributes($attributes);
            $jobOffer->save();

            if ($jobOffer->shouldBeSearchable()) {
                $jobOffer->searchable();
            }

            $archivedOffer->delete();

            return $jobOffer;
        });
    }

    /**
     * Sprawdź czy oferta może być zarchiwizowana
     */
    public function canArchive(JobOffer $jobOffer): bool
    {
        // Możesz dodać własną logikę walidacji
        return !in_array($jobOffer->status?->value ?? $jobOffer->status, [
            Status::ARCHIVED, // jeśli masz taki status
        ]);
    }

    /**
     * Zarchiwizuj wygasłe oferty
     */
    public function archiveExpiredOffers(?string $reason = 'expired'): int
    {
        $expiredOffers = JobOffer::where('expiration_date', '<', now())
            ->where('status', Status::ACTIVE)
            ->get();

        $count = 0;
        foreach ($expiredOffers as $offer) {
            $this->archiveOffer($offer, $reason);
            $count++;
        }

        return $count;
    }

    /**
     * Pobierz statystyki archiwum
     */
    public function getArchiveStatistics(): array
    {
        return [
            'total' => ArchivedJobOffer::count(),
            'by_reason' => ArchivedJobOffer::raw(function ($collection) {
                return $collection->aggregate([
                    [
                        '$group' => [
                            '_id' => '$archived_reason',
                            'count' => ['$sum' => 1]
                        ]
                    ]
                ]);
            })->toArray(),
            'last_30_days' => ArchivedJobOffer::where('archived_at', '>=', now()->subDays(30))->count(),
        ];
    }
}
