<?php

namespace MbeGroup\Job\Filament\Resources;

use MbeGroup\Job\Models\JobOffer;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use MbeGroup\Job\Filament\Resources\JobOfferResource\Pages\ListJobOffers;
use MbeGroup\Job\Filament\Resources\JobOfferResource\Pages\CreateJobOffer;
use MbeGroup\Job\Filament\Resources\JobOfferResource\Pages\EditJobOffer;
use Filament\Actions;
use Filament\Forms;
use App\ValueObject\Status;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use Illuminate\Support\Facades\Log;
use App\Helpers\DictionaryHelper;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use MbeGroup\Employer\Models\Employer;
use App\Filament\Components\JobOfferTemplateFormComponents;
use MbeGroup\Job\Contracts\JobOfferArchiveServiceInterface;
use MbeGroup\Job\Contracts\JobOfferCopyServiceInterface;
use App\Filament\Components\OpenStreetMapFormComponents;
use Illuminate\Support\Collection;
use MbeGroup\Job\Contracts\ConnectJobOffersServiceInterface;

class JobOfferResource extends Resource
{
    protected static ?string $model = JobOffer::class;

    public static function getNavigationGroup(): string
    {
        return __('job::filament.navigation.group.label');
    }

    public static function getNavigationLabel(): string
    {
        return __('job::filament.resources.job_offer.navigation_label');
    }

    public static function getModelLabel(): string
    {
        return __('job::filament.resources.job_offer.model_label');
    }

    public static function getPluralModelLabel(): string
    {
        return __('job::filament.resources.job_offer.plural_model_label');
    }

    public static function form(Schema $schema): Schema
    {
        $statusOptions     = Status::getJobOfferLabels();
        $dictionaryOptions = [
            'industry'                    => DictionaryHelper::getHierarchicalNamesByType('industry'),
            'specialization'              => DictionaryHelper::getHierarchicalNamesByType('specialization'),
            'position_level'              => DictionaryHelper::getHierarchicalNamesByType('position_level'),
            'operating_mode'              => DictionaryHelper::getHierarchicalNamesByType('operating_mode'),
            'hourly_rate'                 => DictionaryHelper::getHierarchicalNamesByType('hourly_rate'),
            'employment_type'             => DictionaryHelper::getHierarchicalNamesByType('employment_type'),
            'application_method'          => DictionaryHelper::getSlugsForOptionsByType('application_method'),
            'preferred_study'             => DictionaryHelper::getHierarchicalNamesByType('preferred_study'),
            'technologies_and_tools'      => DictionaryHelper::getHierarchicalNamesByType('technologies_and_tools'),
            'skills_and_methodologies'    => DictionaryHelper::getHierarchicalNamesByType('skills_and_methodologies'),
            'licenses_and_qualifications' => DictionaryHelper::getHierarchicalNamesByType('licenses_and_qualifications'),
            'languages'                   => DictionaryHelper::getHierarchicalNamesByType('languages'),
            'salary_type'                 => DictionaryHelper::getHierarchicalNamesByType('salary_type'),
            'salary_form'                 => DictionaryHelper::getHierarchicalNamesByType('salary_form'),
            'salary_currency'             => DictionaryHelper::getHierarchicalNamesByType('currency'),
            'position'                    => DictionaryHelper::getHierarchicalNamesByType('position'),
            'level_of_advancement'        => DictionaryHelper::getHierarchicalNamesByType('level_of_advancement'),
            'competence_type'             => DictionaryHelper::getSlugsForOptionsByType('competence_type'),
            'workplace_type'              => DictionaryHelper::getHierarchicalNamesByType('workplace_type'),
            'work_system'                 => DictionaryHelper::getHierarchicalNamesByType('work_system'),
            'delegations'                 => DictionaryHelper::getHierarchicalNamesByType('delegations'),
            'contract_duration'           => DictionaryHelper::getHierarchicalNamesByType('contract_duration'),
            'salary_structure'            => DictionaryHelper::getHierarchicalNamesByType('salary_structure'),
        ];

        return $schema
            ->schema([
                Tabs::make('Tabs')
                    ->persistTabInQueryString()
                    ->tabs([
                        Tab::make(__('job::filament.resources.job_offer.form.tabs.basic_info'))
                            ->schema([
                                Forms\Components\Select::make('employer_id')
                                    ->label(__('content::filament.resources.article.form.fields.employer_id'))
                                    ->options(Employer::getSelectOptions())
                                    ->searchable()
                                    ->required()
                                    ->reactive()
                                    ->afterStateUpdated(function ($state, callable $set) {
                                        if (!$state) {
                                            self::resetEmployerFields($set);
                                            return;
                                        }

                                        try {
                                            $employer = Employer::find($state);

                                            if (!$employer) {
                                                self::resetEmployerFields($set);
                                                return;
                                            }

                                            self::setRefreshFrequency($employer, $set);
                                            self::setApplyOptions($employer, $set);

                                        } catch (\Exception $e) {
                                            Log::error('Error fetching data from employer: ' . $e->getMessage());
                                            self::resetEmployerFields($set);
                                        }
                                    })
                                    ->getSearchResultsUsing(fn (string $search) => Employer::searchForSelect($search)),
                                Forms\Components\Select::make('product_statistics')
                                    ->label(__('job::filament.resources.job_offer.form.fields.product_statistics_id'))
                                    ->options(function (callable $get) {
                                        $employerId = $get('employer_id');
                                        if (!$employerId) {
                                            return [];
                                        }
                                        try {
                                            $products = \MbeGroup\Product\Models\ProductStatistics::where('employer_id', $employerId)
                                                ->where('status', Status::ACTIVE)
                                                ->where('quantity', '>', 0)
                                                ->with(['product' => function ($query) {
                                                    $query->where('category', 'product-job-offers');
                                                }])
                                                ->get()
                                                ->mapWithKeys(fn ($stat) => [
                                                    (string) $stat->id => ($stat->product?->name ?? 'N/A') . ' (' . $stat->created_at?->format('Y-m-d H:i') . ')'
                                                ])
                                                ->toArray();

                                            return $products;
                                        } catch (\Exception $e) {
                                            Log::error('Error fetching products: ' . $e->getMessage());
                                            return [];
                                        }
                                    })
                                    ->getOptionLabelUsing(function ($value): ?string {
                                        try {
                                            $productStatistics = \MbeGroup\Product\Models\ProductStatistics::find($value);
                                            if ($productStatistics && $productStatistics->product) {
                                                return $productStatistics->product->name . ' (' . $productStatistics->created_at?->format('Y-m-d H:i') . ')';
                                            }
                                        } catch (\Exception $e) {
                                            Log::error('Error fetching product label: ' . $e->getMessage());
                                        }
                                        return null;
                                    })
                                    ->searchable()
                                    ->live()
                                    ->afterStateUpdated(function ($state, callable $set) {
                                        if (!$state) {
                                            $set('expiration_date', now()->addDays(1));
                                            return;
                                        }
                                        try {
                                            $productStatistics = \MbeGroup\Product\Models\ProductStatistics::find($state);
                                            if ($productStatistics && $productStatistics->product) {
                                                $days = $productStatistics->product->product_life_days ?? 1;
                                                $set('expiration_date', now()->addDays($days));
                                            }
                                        } catch (\Exception $e) {
                                            Log::error('Error updating expiration_date from product_statistics: ' . $e->getMessage());
                                            $set('expiration_date', now()->addDays(1));
                                        }
                                    })
                                    ->required(fn (callable $get) => $get('status') === Status::ACTIVE),
                                Forms\Components\TextInput::make('title')
                                    ->label(__('job::filament.resources.job_offer.form.fields.title'))
                                    ->required()
                                    ->maxLength(255)
                                    ->live(onBlur: true)
                                    ->afterStateUpdated(function (string $operation, $state, callable $set) {
                                        if ($operation !== 'create') {
                                            return;
                                        }
                                        $set('slug', \Illuminate\Support\Str::slug($state));
                                    }),
                                Forms\Components\TextInput::make('slug')
                                    ->label(__('job::filament.resources.job_offer.form.fields.slug'))
                                    ->maxLength(255)
                                    ->disabled(fn (string $operation): bool => $operation === 'edit')
                                    ->dehydrated(fn (string $operation): bool => $operation === 'create'),
                                Forms\Components\TextInput::make('reference_number')
                                    ->label(__('job::filament.resources.job_offer.form.fields.reference_number')),
                                Forms\Components\Select::make('application_method.apply_options.application_method')
                                    ->label(__('job::filament.resources.job_offer.form.fields.application_method'))
                                    ->options($dictionaryOptions['application_method'])
                                    ->allowHtml()
                                    ->searchable()
                                    ->required()
                                    ->live()
                                    ->afterStateUpdated(function ($state, callable $set) {
                                        if ($state !== 'email') {
                                            $set('application_method.apply_options.application_email', null);
                                        }
                                        if ($state !== 'phone') {
                                            $set('application_method.apply_options.application_phone', null);
                                        }
                                        if ($state !== 'url') {
                                            $set('application_method.apply_options.application_url', null);
                                        }
                                    }),
                                Forms\Components\TextInput::make('application_method.apply_options.application_link_reference')
                                    ->label(__('job::filament.resources.job_offer.form.fields.application_link_reference'))
                                    ->url()
                                    ->visible(fn (callable $get) => $get('application_method.apply_options.application_method') === 'employer-link'),
                                Forms\Components\TextInput::make('application_method.apply_options.application_link_reference_value')
                                    ->label(__('job::filament.resources.job_offer.form.fields.application_link_reference_value'))
                                    ->visible(fn (callable $get) => $get('application_method.apply_options.application_method') === 'employer-link'),
                                Forms\Components\TextInput::make('application_method.apply_options.application_email')
                                    ->label(__('job::filament.resources.job_offer.form.fields.application_email'))
                                    ->email()
                                    ->visible(fn (callable $get) => $get('application_method.apply_options.application_method') === 'form-application')
                                    ->maxLength(255),
                                Forms\Components\Checkbox::make('application_method.apply_options.application_remote_recruitment')
                                    ->label(__('job::filament.resources.job_offer.form.fields.application_remote_recruitment'))
                                    ->visible(fn (callable $get) => $get('application_method.apply_options.application_method') === 'form-application')
                                    ->default(false),
                                ]),
                    Tab::make(__('job::filament.resources.job_offer.form.tabs.category'))
                        ->schema([
                            OpenStreetMapFormComponents::getOpenStreetMapFields(),
                            Forms\Components\Select::make('industry_ids')
                                ->label(__('job::filament.resources.job_offer.form.fields.industry'))
                                ->options($dictionaryOptions['industry'])
                                ->allowHtml()
                                ->multiple()
                                ->searchable()
                                ->required(),
                            Forms\Components\Select::make('specialization_ids')
                                ->label(__('job::filament.resources.job_offer.form.fields.specialization'))
                                ->options($dictionaryOptions['specialization'])
                                ->allowHtml()
                                ->searchable()
                                ->multiple()
                                ->required(),
                            Forms\Components\Select::make('position_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.position'))
                                ->options($dictionaryOptions['position'])
                                ->allowHtml()
                                ->searchable(),
                            Forms\Components\Select::make('position_level_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.position_level'))
                                ->options($dictionaryOptions['position_level'])
                                ->allowHtml()
                                ->searchable(),
                            Forms\Components\Select::make('operating_mode_ids')
                                ->label(__('job::filament.resources.job_offer.form.fields.operating_mode'))
                                ->options($dictionaryOptions['operating_mode'])
                                ->allowHtml()
                                ->multiple()
                                ->searchable(),
                            Forms\Components\Select::make('hourly_rate_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.hourly_rate'))
                                ->options($dictionaryOptions['hourly_rate'])
                                ->allowHtml()
                                ->searchable(),
                            Forms\Components\Select::make('preferred_study_ids')
                                ->label(__('job::filament.resources.job_offer.form.fields.preferred_study'))
                                ->options($dictionaryOptions['preferred_study'])
                                ->allowHtml()
                                ->searchable()
                                ->multiple(),
                            Forms\Components\Select::make('workplace_type_ids')
                                ->label(__('job::filament.resources.job_offer.form.fields.workplace_type'))
                                ->options($dictionaryOptions['workplace_type'])
                                ->allowHtml()
                                ->searchable()
                                ->multiple(),
                            Forms\Components\Select::make('work_system_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.work_system'))
                                ->options($dictionaryOptions['work_system'])
                                ->allowHtml()
                                ->searchable(),
                             Forms\Components\Select::make('delegations_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.delegations'))
                                ->options($dictionaryOptions['delegations'])
                                ->allowHtml()
                                ->searchable(),
                            Forms\Components\Select::make('contract_duration_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.contract_duration'))
                                ->options($dictionaryOptions['contract_duration'])
                                ->allowHtml()
                                ->searchable(),
                            Forms\Components\Select::make('salary_structure_ids')
                                ->label(__('job::filament.resources.job_offer.form.fields.salary_structure'))
                                ->options($dictionaryOptions['salary_structure'])
                                ->allowHtml()
                                ->searchable()
                                ->multiple(),
                            Forms\Components\Checkbox::make('is_for_technicians')
                                ->label(__('job::filament.resources.job_offer.form.fields.is_for_technicians')),
                            Forms\Components\Checkbox::make('is_remote_recruitment')
                                ->label(__('job::filament.resources.job_offer.form.fields.is_remote_recruitment')),
                            Forms\Components\Checkbox::make('has_team_management')
                                ->label(__('job::filament.resources.job_offer.form.fields.has_team_management')),
                            Forms\Components\Checkbox::make('has_driving_license')
                                ->label(__('job::filament.resources.job_offer.form.fields.has_driving_license')),
                            Forms\Components\Checkbox::make('no_experience')
                                ->label(__('job::filament.resources.job_offer.form.fields.no_experience')),
                            Forms\Components\Checkbox::make('has_company_car')
                                ->label(__('job::filament.resources.job_offer.form.fields.has_company_car')),
                        ]),
                    Tab::make(__('job::filament.resources.job_offer.form.tabs.competences'))
                        ->schema(function () use ($dictionaryOptions) {
                            $repeaters = [];

                            foreach ($dictionaryOptions['competence_type'] as $slug => $label) {
                                $competenceOptions = DictionaryHelper::getSlugsForOptionsByParentSlug($slug);
                                $repeaters[]       = Forms\Components\Repeater::make('competences.' . $slug)
                                    ->label($label)
                                    ->lazy()
                                    ->defaultItems(0)
                                    ->collapsible()
                                    ->collapsed()
                                    ->addActionLabel(__('job::filament.resources.job_offer.form.fields.add_competence'))
                                    ->schema([
                                        Forms\Components\Select::make('key')
                                            ->label(__('job::filament.resources.job_offer.form.fields.competence_name'))
                                            ->options($competenceOptions)
                                            ->allowHtml()
                                            ->searchable()
                                            ->required(),
                                        Forms\Components\Select::make('value')
                                            ->label(__('job::filament.resources.job_offer.form.fields.competence_level'))
                                            ->options($dictionaryOptions['level_of_advancement'] ?? [])
                                            ->allowHtml()
                                            ->searchable(),
                                    ]);
                            }

                            $repeaters[] = Section::make(__('job::filament.resources.job_offer.form.sections.nice_to_have'))
                                ->schema([
                                    Forms\Components\Select::make('nice_to_have_language_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.languages'))
                                        ->options($dictionaryOptions['languages'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                    Forms\Components\Select::make('nice_to_have_technologies_and_tools_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.technologies_and_tools'))
                                        ->options($dictionaryOptions['technologies_and_tools'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                    Forms\Components\Select::make('nice_to_have_skills_and_methodologies_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.skills_and_methodologies'))
                                        ->options($dictionaryOptions['skills_and_methodologies'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                    Forms\Components\Select::make('nice_to_have_licenses_and_qualifications_ids')
                                        ->label(__('job::filament.resources.job_offer.form.fields.licenses_and_qualifications'))
                                        ->options($dictionaryOptions['licenses_and_qualifications'])
                                        ->allowHtml()
                                        ->searchable()
                                        ->multiple(),
                                ]);

                            return $repeaters;
                        }),
                    Tab::make(__('job::filament.resources.job_offer.form.tabs.salary_and_contracts'))
                        ->schema([
                            Forms\Components\Repeater::make('salary')
                            ->label(__('job::filament.resources.job_offer.form.fields.contracts'))
                            ->lazy()
                            ->addActionLabel(__('job::filament.resources.job_offer.form.fields.add_contract'))
                            ->required()
                            ->schema([
                                Forms\Components\Select::make('salary_employment_type_id')
                                ->label(__('job::filament.resources.job_offer.form.fields.salary_employment_type'))
                                ->options($dictionaryOptions['employment_type'])
                                ->allowHtml()
                                ->required()
                                ->searchable(),
                                Section::make(__('job::filament.resources.job_offer.form.fields.salary'))
                                    ->schema([
                                        Forms\Components\Select::make('salary_type')
                                            ->label(__('job::filament.resources.job_offer.form.fields.salary_type'))
                                            ->options($dictionaryOptions['salary_type'])
                                            ->allowHtml(),
                                        Forms\Components\Select::make('salary_form')
                                            ->label(__('job::filament.resources.job_offer.form.fields.salary_form'))
                                            ->options($dictionaryOptions['salary_form'])
                                            ->allowHtml(),
                                        Forms\Components\TextInput::make('salary_from')
                                            ->label(__('job::filament.resources.job_offer.form.fields.salary_from'))
                                            ->numeric(),
                                        Forms\Components\TextInput::make('salary_to')
                                            ->label(__('job::filament.resources.job_offer.form.fields.salary_to'))
                                            ->numeric(),
                                        Forms\Components\Select::make('salary_currency')
                                            ->label(__('job::filament.resources.job_offer.form.fields.salary_currency'))
                                            ->options($dictionaryOptions['salary_currency'])
                                            ->allowHtml(),
                                        Forms\Components\TextInput::make('salary_note')
                                            ->label(__('job::filament.resources.job_offer.form.fields.salary_note')),
                                        Forms\Components\Checkbox::make('salary_range_is_visible')
                                            ->label(__('job::filament.resources.job_offer.form.fields.salary_range_is_visible'))
                                            ->default(true),
                                    ])
                                    ->collapsed()
                                    ->collapsible(),
                            ]),
                            Section::make(__('job::filament.resources.job_offer.form.fields.salary_for_search'))
                                ->schema([
                                    Grid::make(2)
                                        ->schema([
                                            Forms\Components\TextInput::make('salary_for_search.from')
                                                ->label(__('job::filament.resources.job_offer.form.fields.salary_for_search_from'))
                                                ->disabled(),
                                            Forms\Components\TextInput::make('salary_for_search.to')
                                                ->label(__('job::filament.resources.job_offer.form.fields.salary_for_search_to'))
                                                ->disabled(),
                                        ]),
                                ]),
                        ]),
                    Tab::make(__('job::filament.resources.job_offer.form.tabs.content'))
                        ->schema(
                            function (callable $get) {
                                return JobOfferTemplateFormComponents::getJobOfferTemplate($get);
                            }),
                    Tab::make(__('job::filament.resources.job_offer.form.tabs.publication_settings'))
                        ->schema([
                            Forms\Components\Select::make('status')
                                ->label(__('job::filament.resources.job_offer.form.fields.status'))
                                ->options($statusOptions)
                                ->required()
                                ->live()
                                ->default(Status::DRAFT),
                            Forms\Components\DateTimePicker::make('created_at')
                                ->label(__('job::filament.resources.job_offer.form.fields.created_at'))
                                ->default(now())
                                ->required(),
                            Forms\Components\DateTimePicker::make('updated_at')
                                ->label(__('job::filament.resources.job_offer.form.fields.updated_at'))
                                ->default(fn (string $operation) => $operation === 'create' ? now() : null),
                            Forms\Components\DateTimePicker::make('publication_date')
                                ->label(__('job::filament.resources.job_offer.form.fields.publication_date'))
                                ->default(now()),
                            Forms\Components\DateTimePicker::make('expiration_date')
                                ->label(__('job::filament.resources.job_offer.form.fields.expiration_date'))
                                ->default(function (callable $get) {
                                    $productStatisticsId = $get('product_statistics');
                                    if (!$productStatisticsId) {
                                        return now()->addDays(1);
                                    }
                                    try {
                                        $productStatistics = \MbeGroup\Product\Models\ProductStatistics::find($productStatisticsId);
                                        if ($productStatistics && $productStatistics->product) {
                                            $days = $productStatistics->product->product_life_days ?? 1;
                                            return now()->addDays($days);
                                        }
                                    } catch (\Exception $e) {
                                        Log::error('Error fetching product_life_days: ' . $e->getMessage());
                                    }
                                    return now()->addDays(1);
                                })
                                ->readOnly(true),
                            Forms\Components\Checkbox::make('is_top_offer')
                                ->label(__('job::filament.resources.job_offer.form.fields.is_top_offer')),
                            Forms\Components\Select::make('refresh_frequency')
                                ->label(__('job::filament.resources.job_offer.form.fields.refresh_frequency'))
                                ->options([
                                    1 => 1, 7 => 7, 14 => 14
                                ])
                                ->default('none')
                        ])

                    ]),
            ])->columns(1);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->defaultSort('created_at', 'desc')
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label('ID')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('public_id')
                    ->label('Public ID')
                    ->searchable()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('title')
                    ->label(__('job::filament.resources.job_offer.table.columns.title'))
                    ->searchable()
                    ->limit(50)
                    ->toggleable()
                    ->icon(fn ($record) => ($record->connection_id && $record->isConnected()) ? 'heroicon-o-link' : null)
                    ->iconColor(function ($record) {
                        if (!$record->connection_id || !$record->isConnected()) {
                            return null;
                        }

                        $colors = [
                            'blue', 'green', 'red', 'yellow', 'purple', 'pink',
                            'indigo', 'orange', 'teal', 'cyan', 'lime', 'amber',
                            'emerald', 'violet', 'fuchsia', 'rose', 'sky', 'slate'
                        ];

                        $hash       = crc32($record->connection_id);
                        $colorIndex = abs($hash) % count($colors);

                        return $colors[$colorIndex];
                    }),
                Tables\Columns\TextColumn::make('employer.name')
                    ->label(__('content::filament.resources.content.table.columns.employer'))
                    ->placeholder('Brak danych')
                    ->searchable()
                    ->toggleable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('locations')
                    ->label(__('job::filament.resources.job_offer.table.columns.location'))
                    ->formatStateUsing(function ($state) {

                        return $state['location_name'] ?? 'Brak danych';
                    })
                    ->searchable()
                    ->toggleable(),
                Tables\Columns\IconColumn::make('status')
                    ->label(__('job::filament.resources.job_offer.table.columns.status'))
                    ->state(fn ($record) => Status::int($record->status))
                    ->icon(fn ($state) => Status::icon($state))
                    ->color(fn ($state) => Status::color($state))
                    ->tooltip(fn ($state) => Status::label($state))
                    ->sortable()
                    ->toggleable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label(__('job::filament.resources.job_offer.table.columns.created_at'))
                    ->dateTime()
                    ->toggleable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('publication_date')
                    ->label(__('job::filament.resources.job_offer.table.columns.publication_date'))
                    ->dateTime()
                    ->toggleable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('expiration_date')
                    ->label(__('job::filament.resources.job_offer.table.columns.expiration_date'))
                    ->dateTime()
                    ->toggleable()
                    ->sortable(),
            ])
            ->filters([
                      //
            ])
            ->recordActions([
                Actions\EditAction::make()
                    ->iconButton()
                    ->color('info'),
                Actions\ViewAction::make()
                    ->iconButton()
                    ->color('gray'),
                Actions\Action::make('openLink')
                    ->iconButton()
                    ->icon('heroicon-o-arrow-top-right-on-square')
                    ->color('success')
                    ->url(fn (JobOffer $record) => url('/oferta-pracy/' . $record->slug . ',' . $record->id))
                    ->openUrlInNewTab(),
                Actions\Action::make('copy')
                    ->iconButton()
                    ->icon('heroicon-o-clipboard-document')
                    ->label(__('job::filament.resources.job_offer.actions.copy.label'))
                    ->color('gray')
                    ->action(function (JobOffer $record, Actions\Action $action) {
                        $newOffer = app(JobOfferCopyServiceInterface::class)->copyOffer($record);
                        $action->redirect(JobOfferResource::getUrl('edit', ['record' => $newOffer->id]));
                    }),
                Actions\Action::make('archive')
                    ->iconButton()
                    ->icon('heroicon-o-archive-box-x-mark')
                    ->modalHeading(__('job::filament.resources.job_offer.actions.archive.modal.heading'))
                    ->requiresConfirmation()
                    ->color('warning')
                    ->modalDescription(__('job::filament.resources.job_offer.actions.archive.modal.description'))
                    ->action(function (JobOffer $record, array $data) {
                        app(JobOfferArchiveServiceInterface::class)->archiveOffer($record, 'Archived via Filament Admin Panel');
                    }),
                Actions\DeleteAction::make()
                    ->iconButton()
                    ->color('danger'),
            ])
            ->toolbarActions([
                Actions\BulkActionGroup::make([
                    Actions\DeleteBulkAction::make(),
                    Actions\BulkAction::make('archive')
                        ->label(__('job::filament.resources.job_offer.actions.bulk_archive.label'))
                        ->icon('heroicon-o-archive-box-x-mark')
                        ->requiresConfirmation()
                        ->color('warning')
                        ->modalHeading(__('job::filament.resources.job_offer.actions.bulk_archive.modal.heading'))
                        ->modalDescription(__('job::filament.resources.job_offer.actions.bulk_archive.modal.description'))
                        ->action(function (Collection $records, array $data) {
                            foreach ($records as $record) {
                                app(JobOfferArchiveServiceInterface::class)->archiveOffer($record, 'Archived via Filament Admin Panel - Bulk Action');
                            }
                        }),
                    Actions\BulkAction::make('connect')
                        ->label(__('job::filament.resources.job_offer.actions.bulk_connect.label'))
                        ->icon('heroicon-o-link')
                        ->requiresConfirmation()
                        ->color('primary')
                        ->modalHeading(__('job::filament.resources.job_offer.actions.bulk_connect.modal.heading'))
                        ->modalDescription(__('job::filament.resources.job_offer.actions.bulk_connect.modal.description'))
                        ->action(function (Collection $records, Actions\BulkAction $action) {
                            $titles = $records->pluck('title')->unique();

                            if ($titles->count() > 1) {
                                \Filament\Notifications\Notification::make()
                                    ->warning()
                                    ->title('Nie można połączyć ofert')
                                    ->body('Wybrane oferty mają różne tytuły. Możesz połączyć tylko oferty o identycznym tytule.')
                                    ->send();

                                $action->halt();
                                return;
                            }

                            $employers = $records->pluck('employer_id')->unique();

                            if ($employers->count() > 1) {
                                \Filament\Notifications\Notification::make()
                                    ->warning()
                                    ->title('Nie można połączyć ofert')
                                    ->body('Wybrane oferty należą do różnych pracodawców. Możesz połączyć tylko oferty tego samego pracodawcy.')
                                    ->send();

                                $action->halt();
                                return;
                            }

                            $ids = $records->pluck('id')->toArray();
                            app(ConnectJobOffersServiceInterface::class)->connectOffersByIds($ids);

                            \Filament\Notifications\Notification::make()
                                ->success()
                                ->title('Oferty połączone')
                                ->body('Połączono ' . $records->count() . ' ofert.')
                                ->send();
                        }),
                ]),
            ])
            ->description('Version: ' . \Composer\InstalledVersions::getVersion('mbegroup/job'));
    }

    public static function getRelations(): array
    {
        return [
                  //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index'  => ListJobOffers::route('/'),
            'create' => CreateJobOffer::route('/create'),
            'edit'   => EditJobOffer::route('/{record}/edit'),
        ];
    }

    protected static function setRefreshFrequency(Employer $employer, callable $set): void
    {
        $refreshFrequency = $employer->publication_settings['job_offers_refresh']['refresh_frequency'] ?? null;

        if ($refreshFrequency) {
            $slug = DictionaryHelper::getSlugById($refreshFrequency);
            $set('refresh_frequency', $slug);
        } else {
            $set('refresh_frequency', 'none');
        }
    }

    protected static function setApplyOptions(Employer $employer, callable $set): void
    {
        $applyOptions = $employer->publication_settings['apply_options'] ?? null;

        $fields = [
            'application_method',
            'application_link_reference',
            'application_link_reference_value',
            'application_email',
            'application_remote_recruitment',
        ];

        foreach ($fields as $field) {
            $value = $applyOptions[$field] ?? null;
            $set("application_method.apply_options.{$field}", $value);
        }
    }

    protected static function resetEmployerFields(callable $set): void
    {
        $set('refresh_frequency', 'none');

        $fields = [
            'application_method',
            'application_link_reference',
            'application_link_reference_value',
            'application_email',
            'application_remote_recruitment',
        ];

        foreach ($fields as $field) {
            $set("application_method.apply_options.{$field}", null);
        }
    }
}
