<?php
namespace MbeGroup\Job\Models;

use Illuminate\Support\Str;
use MongoDB\Laravel\Eloquent\Model;
use Laravel\Scout\Searchable;
use App\ValueObject\Status;
use MbeGroup\Employer\Models\Employer;
use MbeGroup\Product\Models\ProductStatistics;
use App\Traits\HasSeoFields;
use App\Traits\HasDictionaryRelations;
use App\Models\Counter;

class JobOffer extends Model
{
    use Searchable, HasSeoFields, HasDictionaryRelations;

    protected $connection = 'mongodb';
    protected $collection = 'job_offers';
    protected $fillable   = [
        'public_id',
        'employer_id',
        'product_statistics',
        'title',
        'slug',
        'reference_number',
        'status',
        'locations',
        'industry_ids',
        'specialization_ids',
        'position_id',
        'position_level_id',
        'is_for_technicians',
        'operating_mode_ids',
        'is_remote_recruitment',
        'employment_type_id',
        'hourly_rate_id',
        'preferred_study_ids',
        'workplace_type_ids',
        'work_system_id',
        'delegations_id',
        'contract_duration_id',
        'salary_structure_ids',
        'has_team_management',
        'has_driving_license',
        'no_experience',
        'has_company_car',
        'nice_to_have_language_ids',
        'nice_to_have_technologies_and_tools_ids',
        'nice_to_have_skills_and_methodologies_ids',
        'nice_to_have_licenses_and_qualifications_ids',
        'competences',
        'job_offer_template_name',
        'template_variables',
        'application_method',
        'publication_date',
        'expiration_date',
        'created_by',
        'updated_by',
        'refresh_frequency',
        'is_top_offer',
        'template',
        'salary',
        'salary_for_search',
        'connection_id',
        'provider_id',
        'template_benefits',
        'template_recruitment_stages',
    ];

    protected $casts = [
        'status'           => Status::class,
        'locations'        => 'array',
        'publication_date' => 'datetime',
        'expiration_date'  => 'datetime',
    ];

    protected static function boot(): void
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->public_id)) {
                $model->public_id = static::generatePublicId();
            }

            if (!empty($model->slug)) {
                $model->slug = static::generateUniqueSlug($model->slug);
            }
        });
    }

    private static function generatePublicId(): string
    {
        $collectionName = (new static())->getTable();
        $next           = Counter::getNext($collectionName);

        return str_pad((string) $next, 6, '0', STR_PAD_LEFT);
    }

    private static function generateUniqueSlug(string $slug): string
    {
        $original = $slug;
        $counter  = 2;

        while (static::where('slug', $slug)->exists()) {
            $slug = $original . '-' . $counter++;
        }

        return $slug;
    }

    public function employer()
    {
        return $this->belongsTo(Employer::class, 'employer_id');
    }

    public function productStatistics()
    {
        return $this->belongsTo(ProductStatistics::class, 'product_statistics');
    }

    public function isConnected(): bool
    {
        if (empty($this->connection_id)) {
            return false;
        }

        $count = static::where('connection_id', $this->connection_id)
            ->where('_id', '!=', $this->_id)
            ->count();

        return $count > 0;
    }

    public function connect(string $connectionId): void
    {
        $this->connection_id = $connectionId;
        $this->save();
    }

    public function toSearchableArray()
    {

        if (!$this->relationLoaded('employer')) {
            $this->load('employer');
        }

        $employer = $this->employer;

        $array = [
            'id'          => $this->_id,
            'public_id'   => $this->public_id,
            'employer_id' => $this->employer_id,
            'employer'    => $employer ? [
                'id'              => $employer->_id,
                'name'            => $employer->name ?? null,
                'slug'            => $employer->slug ?? null,
                'company_size_id' => $employer->company_size_id ?? null,
                'company_type_id' => $employer->company_type_id ?? null,
            ] : null,
            'product_statistics' => $this->product_statistics,
            'title'              => $this->title,
            'slug'               => $this->slug,
            'reference_number'   => $this->reference_number,
            'status'             => $this->status?->value ?? $this->status,
            'locations'          => array_map(function($loc) {
                    return [
                        'location_name'       => $loc['location_name'] ?? null,
                        'municipality'        => $loc['municipality'] ?? null,
                        'region'              => $loc['region'] ?? null,
                        'postal_code'         => $loc['postal_code'] ?? null,
                        'status'              => $loc['status'] ?? null,
                        'location_is_default' => $loc['location_is_default'] ?? false,
                        'location'            => [
                            'lat' => (float) ($loc['latitude'] ?? 0),
                            'lon' => (float) ($loc['longitude'] ?? 0)
                        ]
                    ];
                }, $this->locations ?? []),
            'industry_ids'                                 => $this->industry_ids,
            'specialization_ids'                           => $this->specialization_ids,
            'position_id'                                  => $this->position_id,
            'position_level_id'                            => $this->position_level_id,
            'is_for_technicians'                           => $this->is_for_technicians,
            'operating_mode_ids'                           => $this->operating_mode_ids,
            'is_remote_recruitment'                        => $this->is_remote_recruitment,
            'employment_type_id'                           => $this->employment_type_id,
            'hourly_rate_id'                               => $this->hourly_rate_id,
            'preferred_study_ids'                          => $this->preferred_study_ids,
            'workplace_type_ids'                           => $this->workplace_type_ids,
            'work_system_id'                               => $this->work_system_id,
            'delegations_id'                               => $this->delegations_id,
            'contract_duration_id'                         => $this->contract_duration_id,
            'salary_structure_ids'                         => $this->salary_structure_ids,
            'has_team_management'                          => $this->has_team_management,
            'has_driving_license'                          => $this->has_driving_license,
            'no_experience'                                => $this->no_experience,
            'has_company_car'                              => $this->has_company_car,
            'competences'                                  => $this->competences,
            'nice_to_have_language_ids'                    => $this->nice_to_have_language_ids,
            'nice_to_have_technologies_and_tools_ids'      => $this->nice_to_have_technologies_and_tools_ids,
            'nice_to_have_skills_and_methodologies_ids'    => $this->nice_to_have_skills_and_methodologies_ids,
            'nice_to_have_licenses_and_qualifications_ids' => $this->nice_to_have_licenses_and_qualifications_ids,
            'job_offer_template_name'                      => $this->job_offer_template_name,
            'template_variables'                           => $this->template_variables,
            'application_method'                           => $this->application_method,
            'is_top_offer'                                 => $this->is_top_offer,
            'publication_date'                             => $this->publication_date?->timestamp,
            'expiration_date'                              => $this->expiration_date?->timestamp,
            'created_by'                                   => $this->created_by,
            'updated_by'                                   => $this->updated_by,
            'refresh_frequency'                            => $this->refresh_frequency,
            'template'                                     => $this->template,
            'salary'                                       => $this->salary,
            'salary_for_search'                            => $this->salary_for_search,
            'template_benefits'                            => $this->template_benefits,
            'template_recruitment_stages'                  => $this->template_recruitment_stages,
            'created_at'                                   => $this->created_at?->timestamp,
            'updated_at'                                   => $this->updated_at?->timestamp,
        ];

        return $array;
    }

    public function searchableAs()
    {
        return 'job_offers';
    }

    public function shouldBeSearchable()
    {
        return $this->status?->value === Status::ACTIVE || $this->status === Status::ACTIVE;
    }

    public function getScoutKey()
    {
        return $this->_id;
    }

    public function getScoutKeyName()
    {
        return '_id';
    }

    protected static function makeAllSearchableUsing($query)
    {
        return $query->with('employer');
    }

    public function getIndustries()
    {
        return $this->getDictionaryItems($this->industry_ids ?? null);
    }

    public function getSpecializations()
    {
        return $this->getDictionaryItems($this->specialization_ids ?? null);
    }

    public function getPosition()
    {
        return $this->getDictionaryItem($this->position_id ?? null);
    }

    public function getPositionLevel()
    {
        return $this->getDictionaryItem($this->position_level_id ?? null);
    }

    public function getOperatingModes()
    {
        return $this->getDictionaryItems($this->operating_mode_ids ?? null);
    }

    public function getEmploymentTypes()
    {
        return $this->getDictionaryItems($this->employment_type_id ?? null);
    }

    public function getHourlyRate()
    {
        return $this->getDictionaryItem($this->hourly_rate_id ?? null);
    }

    public function getWorkplaceTypes()
    {
        return $this->getDictionaryItems($this->workplace_type_ids ?? null);
    }

    public function getWorkSystem()
    {
        return $this->getDictionaryItem($this->work_system_id ?? null);
    }

    public function getDelegations()
    {
        return $this->getDictionaryItem($this->delegations_id ?? null);
    }

    public function getContractDuration()
    {
        return $this->getDictionaryItem($this->contract_duration_id ?? null);
    }

    public function getSalaryStructures()
    {
        return $this->getDictionaryItems($this->salary_structure_ids ?? null);
    }

    public function getPreferredStudies()
    {
        return $this->getDictionaryItems($this->preferred_study_ids ?? null);
    }

    public function getNiceToHaveLanguages()
    {
        return $this->getDictionaryItems($this->nice_to_have_language_ids ?? null);
    }

    public function getNiceToHaveTechnologiesAndTools()
    {
        return $this->getDictionaryItems($this->nice_to_have_technologies_and_tools_ids ?? null);
    }

    public function getNiceToHaveSkillsAndMethodologies()
    {
        return $this->getDictionaryItems($this->nice_to_have_skills_and_methodologies_ids ?? null);
    }

    public function getNiceToHaveLicensesAndQualifications()
    {
        return $this->getDictionaryItems($this->nice_to_have_licenses_and_qualifications_ids ?? null);
    }

    public function getPreparedCompetences()
    {
        $competences = $this->competences;

        if (empty($competences) || !is_array($competences)) {
            return [];
        }

        $categorySlugs = array_keys($competences);
        $itemSlugs     = [];
        $itemIds       = [];

        foreach ($competences as $items) {
            if (!is_array($items)) {
                continue;
            }
            foreach ($items as $item) {
                if (!empty($item['key'])) {
                    $itemSlugs[] = $item['key'];
                }
                if (!empty($item['value'])) {
                    $itemIds[] = (string) $item['value'];
                }
            }
        }

        $allSlugs = array_unique(array_merge($categorySlugs, $itemSlugs));

        $bySlug = \App\Models\Dictionary::select('id', 'name', 'slug')
            ->whereIn('slug', $allSlugs)
            ->get()
            ->keyBy('slug');

        $byId = !empty($itemIds)
            ? \App\Models\Dictionary::select('id', 'name', 'slug')
                ->whereIn('id', $itemIds)
                ->get()
                ->keyBy('id')
            :   collect();

        $result = [];

        foreach ($competences as $categorySlug => $items) {
            $categoryEntry = $bySlug->get($categorySlug);

            $values = [];
            if (is_array($items)) {
                foreach ($items as $item) {
                    $keyEntry   = isset($item['key'])   ? $bySlug->get($item['key'])              : null;
                    $valueEntry = isset($item['value']) ? $byId->get((string) $item['value'])     : null;

                    $values[] = [
                        'key'   => $keyEntry   ? $keyEntry->name   : ($item['key']   ?? null),
                        'value' => $valueEntry ? $valueEntry->name : ($item['value'] ?? null),
                    ];
                }
            }

            $result[$categorySlug] = [
                'name'   => $categoryEntry ? $categoryEntry->name : $categorySlug,
                'values' => $values,
            ];
        }

        return $result;
    }

    public function getPreparedSalary()
    {
        $salary = $this->salary;

        if (empty($salary) || !is_array($salary)) {
            return [];
        }
        $ids = [];
        foreach ($salary as $item) {
            foreach ($item as $value) {
                if (is_int($value) || (is_string($value) && ctype_digit($value))) {
                    $ids[] = (string) $value;
                }
            }
        }

        $byId = !empty($ids)
            ? \App\Models\Dictionary::select('id', 'name', 'slug')
                ->whereIn('id', array_unique($ids))
                ->get()
                ->keyBy('id')
            :   collect();

        return array_map(function ($item) use ($byId) {
            $resolved = $item;
            foreach ($item as $field => $value) {
                if (is_int($value) || (is_string($value) && ctype_digit($value))) {
                    $entry = $byId->get((string) $value);
                    if ($entry) {
                        $resolved[$field] = ['id' => $entry->id, 'name' => $entry->name, 'slug' => $entry->slug];
                    }
                }
            }
            return $resolved;
        }, $salary);
    }
}
