<?php

namespace MbeGroup\Job\Filament\Resources;

use MbeGroup\Job\Models\JobOffer;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use MbeGroup\Job\Filament\Resources\ImportResource\Pages\ListImports;
use MbeGroup\Job\Filament\Resources\ImportResource\Pages\CreateImport;
use MbeGroup\Job\Filament\Resources\ImportResource\Pages\EditImport;
use MbeGroup\Job\Filament\Resources\ImportResource\Pages\ViewImport;
use Filament\Actions;
use Filament\Forms;
use App\ValueObject\Status;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use Illuminate\Support\Facades\Log;
use App\Helpers\DictionaryHelper;
use Tapp\FilamentGoogleAutocomplete\Forms\Components\GoogleAutocomplete;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use MbeGroup\Employer\Models\Employer;
use App\Filament\Components\JobOfferTemplateFormComponents;
use Livewire\Form;
use MbeGroup\Job\Models\Import;

class ImportResource extends Resource
{
    protected static ?string $model = Import::class;

        /**
     * Configure Status field with common hydration and dehydration logic
     */
    private static function configureStatusField(Forms\Components\Select $field): Forms\Components\Select
    {
        return $field
            ->afterStateHydrated(function (Forms\Components\Select $component, $state): void {
                if ($state instanceof Status) {
                    $component->state($state->value);
                }
            })
            ->dehydrateStateUsing(function ($state): int {
                return $state instanceof Status ? $state->value: (int) $state;
            })
            ->rule('integer');
    }

    public static function getNavigationGroup(): string
    {
        return __('job::filament.navigation.group.label');
    }

    public static function getNavigationLabel(): string
    {
        return __('job::filament.resources.import.navigation_label');
    }

    public static function getModelLabel(): string
    {
        return __('job::filament.resources.import.model_label');
    }

    public static function getPluralModelLabel(): string
    {
        return __('job::filament.resources.import.plural_model_label');
    }

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->schema([
                    Forms\Components\TextInput::make('name')
                        ->label(__('job::filament.resources.import.form.fields.name'))
                        ->required()
                        ->maxLength(255),
                    Forms\Components\Select::make('employer_id')
                        ->label(__('job::filament.resources.import.form.fields.employer'))
                        ->relationship('employer', 'name')
                        ->required(),
                    Forms\Components\TextInput::make('provider_type')
                        ->label(__('job::filament.resources.import.form.fields.provider_type'))
                        ->required()
                        ->maxLength(255),
                    Forms\Components\TextInput::make('url')
                        ->label(__('job::filament.resources.import.form.fields.url'))
                        ->required()
                        ->maxLength(2048),
                    Forms\Components\Select::make('template_name')
                        ->label(__('job::filament.resources.import.form.fields.template'))
                        ->options(function (callable $get) {
                            $employerId = $get('employer_id');
                            if (!$employerId) {
                                return [];
                            }
                            $employer = Employer::find($employerId);
                            if (!$employer) {
                                return [];
                            }
                            $templates = $employer->job_offer_template ?? [];
                            $options = [];
                            foreach ($templates as $template) {
                                if (isset($template['job_offer_template_name'])) {
                                    $name = $template['job_offer_template_name'];
                                    $options[$name] = $name;
                                }
                            }
                            return $options;
                        })
                        ->required()
                        ->reactive()
                        ->searchable(),
                    Forms\Components\TimePicker::make('scheduled_time')
                        ->label(__('job::filament.resources.import.form.fields.scheduled_time'))
                        ->required(),
                    Forms\Components\Textarea::make('config')
                        ->label(__('job::filament.resources.import.form.fields.config'))
                        ->rows(10),
                self::configureStatusField(
                    Forms\Components\Select::make('status')
                        ->label(__('job::filament.resources.job_offer.form.fields.status'))
                        ->options(Status::getJobOfferLabels())
                        ->required()
                        ->default(Status::ACTIVE)
                ),
            ])->columns(1);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label(__('job::filament.resources.import.table.columns.id'))
                    ->sortable()
                    ->searchable()
                    ->toggleable(),
                Tables\Columns\TextColumn::make('name')
                    ->label(__('job::filament.resources.import.table.columns.name'))
                    ->sortable()
                    ->searchable(),
                Tables\Columns\IconColumn::make('status')
                    ->label(__('content::filament.resources.content.table.columns.status'))
                    ->state(fn ($record) => Status::int($record->status))
                    ->icon(fn ($state) => Status::icon($state))
                    ->color(fn ($state) => Status::color($state))
                    ->tooltip(fn ($state) => Status::label($state))
                    ->sortable(),
                Tables\Columns\TextColumn::make('employer.name')
                    ->label(__('job::filament.resources.import.table.columns.employer'))
                    ->sortable()
                    ->searchable(),
                Tables\Columns\TextColumn::make('found_count')
                    ->label(__('job::filament.resources.import.table.columns.found_count'))
                    ->getStateUsing(function ($record) {
                        $runsHistory = $record->runs_history ?? [];
                        $lastRun = end($runsHistory);
                        return $lastRun['finded_offers_count'] ?? 0;
                    })
                    ->sortable(),
                Tables\Columns\TextColumn::make('imported_count')
                    ->label(__('job::filament.resources.import.table.columns.imported_count'))
                    ->getStateUsing(function ($record) {
                        $runsHistory = $record->runs_history ?? [];
                        $lastRun = end($runsHistory);
                        return $lastRun['fetched_count'] ?? 0;
                    })
                    ->sortable(),
                Tables\Columns\TextColumn::make('last_run_at')
                    ->label(__('job::filament.resources.import.table.columns.last_run_at'))
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('last_run_status')
                    ->label(__('job::filament.resources.import.table.columns.last_run_status'))
                    ->getStateUsing(function ($record) {
                        $runsHistory = $record->runs_history ?? [];
                        if (empty($runsHistory)) {
                            return null;
                        }
                        $lastRun = end($runsHistory);
                        return $lastRun['status'] ?? null;
                    })
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'success' => 'success',
                        'failed' => 'danger',
                        default => 'gray',
                    })
                    ->sortable(),
                Tables\Columns\TextColumn::make('scheduled_time')
                    ->label(__('job::filament.resources.import.table.columns.scheduled_time'))
                    ->sortable(),
            ])
            ->filters([
                    //
            ])
            ->recordActions([
                Actions\ViewAction::make()
                    ->iconButton(),
                Actions\EditAction::make()
                    ->iconButton(),
                Actions\DeleteAction::make()
                    ->iconButton(),
            ])
            ->toolbarActions([
                Actions\BulkActionGroup::make([
                    Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->description('Version: ' . \Composer\InstalledVersions::getVersion('mbegroup/job'));
    }

    public static function getRelations(): array
    {
        return [
                //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index'  => ListImports::route('/'),
            'create' => CreateImport::route('/create'),
            'edit'   => EditImport::route('/{record}/edit'),
            'view' =>   ViewImport::route('/{record}'),
        ];
    }
}
