<?php

namespace MbeGroup\Job\Filament\Resources\ImportResource\Pages;

use Filament\Actions;
use Filament\Resources\Pages\ViewRecord;
use MbeGroup\Job\Filament\Resources\ImportResource;
use Filament\Support\Enums\FontWeight;
use MbeGroup\Job\Jobs\ProcessImportJob;
use Filament\Notifications\Notification;
use Filament\Schemas\Schema;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Grid;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Components\IconEntry;
use Filament\Infolists\Components\RepeatableEntry;
use Filament\Infolists\Components\KeyValueEntry;
use Filament\Support\Enums\TextSize;
use App\ValueObject\Status;
use Dom\Text;
use MbeGroup\Job\Contracts\JobOfferArchiveServiceInterface;
use Filament\Infolists\Components\Actions\Action;
use Filament\Infolists\Components\ViewEntry;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Support\HtmlString;

class ViewImport extends ViewRecord
{
    protected static string $resource = ImportResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('run_now')
                ->label(__('job::filament.resources.import.pages.view.actions.run_now.label'))
                ->icon('heroicon-o-play')
                ->color('success')
                ->requiresConfirmation()
                ->modalHeading(__('job::filament.resources.import.pages.view.actions.run_now.modal_heading'))
                ->modalDescription(fn () => __("job::filament.resources.import.pages.view.actions.run_now.modal_description", ['name' => $this->record->name]))
                ->action(function () {
                    ProcessImportJob::dispatch($this->record->_id, app(JobOfferArchiveServiceInterface::class));

                    Notification::make()
                        ->title(__('job::filament.resources.import.pages.view.actions.run_now.notifications.queued.title'))
                        ->body(__('job::filament.resources.import.pages.view.actions.run_now.notifications.queued.body', ['name' => $this->record->name]))
                        ->success()
                        ->send();
                }),

            Actions\EditAction::make(),
            Actions\DeleteAction::make(),
        ];
    }

    public function infolist(Schema $schema): Schema
    {
        return $schema
            ->schema([
                Section::make(__('job::filament.resources.import.form.sections.configuration'))
                    ->schema([
                        Grid::make(4)
                            ->schema([
                                TextEntry::make('name')
                                    ->label(__('job::filament.resources.import.form.fields.name'))
                                    ->weight(FontWeight::Bold)
                                    ->size(TextSize::Medium),
                                TextEntry::make('employer.name')
                                    ->label(__('job::filament.resources.import.form.fields.employer'))
                                    ->placeholder(__('job::filament.resources.import.form.fields.employer_placeholder'))
                                    ->size(TextSize::Large),
                                IconEntry::make('status')
                                    ->label(__('job::filament.resources.import.form.fields.status'))
                                    ->state(fn ($record) => Status::int($record->status))
                                    ->icon(fn ($state) => Status::icon($state))
                                    ->color(fn ($state) => Status::color($state))
                                    ->tooltip(fn ($state) => Status::label($state)),
                                TextEntry::make('is_running')
                                    ->label(__('job::filament.resources.import.form.fields.is_running'))
                                    ->badge()
                                    ->state(fn ($record) => $record->is_running ?? false)
                                    ->formatStateUsing(fn ($state) => $state
                                        ? __('job::filament.resources.import.form.fields.is_running_yes')
                                        : __('job::filament.resources.import.form.fields.is_running_no')
                                    )
                                    ->color(fn ($state) => $state ? 'warning' : 'gray')
                                    ->icon(fn ($state) => $state ? 'heroicon-o-arrow-path' : 'heroicon-o-check-circle'),
                            ]),
                        Grid::make(3)
                            ->schema([
                                TextEntry::make('scheduled_time')
                                    ->label(__('job::filament.resources.import.form.fields.scheduled_time'))
                                    ->icon('heroicon-o-clock'),

                                TextEntry::make('last_run_at')
                                    ->label(__('job::filament.resources.import.form.fields.last_run_at'))
                                    ->icon('heroicon-o-calendar')
                                    ->dateTime()
                                    ->placeholder('--'),

                                TextEntry::make('template_name')
                                    ->label(__('job::filament.resources.import.form.fields.template'))
                                    ->placeholder(__('job::filament.resources.import.form.fields.template')),
                            ]),
                        Grid::make(1)
                            ->schema([
                                TextEntry::make('url')
                                    ->label(__('job::filament.resources.import.form.fields.url'))
                                    ->icon('heroicon-o-link')
                                    ->copyable()
                                    ->columnSpanFull(),
                            ]),
                        Grid::make(2)
                            ->schema([
                                TextEntry::make('provider_type')
                                    ->label(__('job::filament.resources.import.form.fields.provider'))
                                    ->badge()
                                    ->formatStateUsing(fn ($state) =>
                                        \MbeGroup\Job\Providers\ImportProviders\ProviderFactory::getLabel($state) ?? $state
                                    ),
                                TextEntry::make('id')
                                    ->label(__('job::filament.resources.import.form.fields.id'))
                                    ->copyable(),
                            ]),
                    ])
                    ->collapsible(),
                Section::make(__('job::filament.resources.import.form.sections.statistics'))
                    ->schema([
                        Grid::make(4)
                            ->schema([
                                TextEntry::make('id')
                                    ->label(__('job::filament.resources.import.form.fields.total_runs'))
                                    ->state(function ($record) {
                                        $runs = $record->runs_history;

                                        if (is_string($runs)) {
                                            $runs = json_decode($runs, true);
                                        }

                                        if (is_array($runs)) {
                                            return count($runs);
                                        }

                                        return 0;
                                    })
                                    ->icon('heroicon-o-arrow-path')
                                    ->size(TextSize::Large)
                                    ->weight(FontWeight::Bold),

                                TextEntry::make('name')
                                    ->label(__('job::filament.resources.import.form.fields.success_count'))
                                    ->state(function ($record) {
                                        $runs = $record->runs_history;

                                        if (is_string($runs)) {
                                            $runs = json_decode($runs, true);
                                        }

                                        if (!is_array($runs) || empty($runs)) return 0;

                                        return collect($runs)->filter(fn($run) =>
                                            ($run['status'] ?? null) === 'success'
                                        )->count();
                                    })
                                    ->icon('heroicon-o-check-circle')
                                    ->color('success')
                                    ->size(TextSize::Large)
                                    ->weight(FontWeight::Bold),

                                TextEntry::make('url')
                                    ->label(__('job::filament.resources.import.form.fields.failure_count'))
                                    ->state(function ($record) {
                                        $runs = $record->runs_history;

                                        if (is_string($runs)) {
                                            $runs = json_decode($runs, true);
                                        }

                                        if (!is_array($runs) || empty($runs)) return 0;

                                        return collect($runs)->filter(fn($run) =>
                                            ($run['status'] ?? null) === 'failed'
                                        )->count();
                                    })
                                    ->icon('heroicon-o-x-circle')
                                    ->color('danger')
                                    ->size(TextSize::Large)
                                    ->weight(FontWeight::Bold),

                                TextEntry::make('status')
                                    ->label(__('job::filament.resources.import.form.fields.last_status'))
                                    ->badge()
                                    ->size(TextSize::Large)
                                    ->state(function ($record) {
                                        $runs = $record->runs_history;

                                        if (is_string($runs)) {
                                            $runs = json_decode($runs, true);
                                        }

                                        if (!is_array($runs) || empty($runs)) return 'Pending';

                                        $lastRun = end($runs);
                                        return ucfirst($lastRun['status'] ?? 'pending');
                                    })
                                    ->color(function ($record) {
                                        $runs = $record->runs_history;

                                        if (is_string($runs)) {
                                            $runs = json_decode($runs, true);
                                        }

                                        if (!is_array($runs) || empty($runs)) return 'gray';

                                        $lastRun = end($runs);
                                        return match ($lastRun['status'] ?? null) {
                                            'success' => 'success',
                                            'failed' => 'danger',
                                            default => 'gray',
                                        };
                                    }),
                            ]),
                    ]),
                Section::make(__('job::filament.resources.import.form.sections.last_run_details'))
                    ->schema([
                        Grid::make(5)
                            ->schema([
                                TextEntry::make('finded_offers_count')
                                    ->label(__('job::filament.resources.import.form.fields.finded_offers_count'))
                                    ->state(fn ($record) => $record->getLastRun()['finded_offers_count'] ?? '-')
                                    ->color('gray')
                                    ->icon('heroicon-o-magnifying-glass-circle'),
                                TextEntry::make('fetched_count')
                                    ->label(__('job::filament.resources.import.form.fields.fetched_count'))
                                    ->state(fn ($record) => $record->getLastRun()['fetched_count'] ?? '-')
                                    ->icon('heroicon-o-arrow-down-tray'),
                                TextEntry::make('new_count')
                                    ->label(__('job::filament.resources.import.form.fields.new_count'))
                                    ->state(fn ($record) => $record->getLastRun()['new_count'] ?? '-')
                                    ->color('success'),
                                TextEntry::make('restored_count')
                                    ->label(__('job::filament.resources.import.form.fields.restored_count'))
                                    ->state(fn ($record) => $record->getLastRun()['restored_count'] ?? '-')
                                    ->color('warning'),
                                TextEntry::make('active_count')
                                    ->label(__('job::filament.resources.import.form.fields.active_count'))
                                    ->state(fn ($record) => $record->getLastRun()['active_count'] ?? '-')
                                    ->color('info'),
                                TextEntry::make('archived_count')
                                    ->label(__('job::filament.resources.import.form.fields.archived_count'))
                                    ->state(fn ($record) => $record->getLastRun()['archived_count'] ?? '-')
                                    ->color('gray'),
                            ]),
                        TextEntry::make('updated_at')
                            ->label(__('job::filament.resources.import.form.fields.duration_seconds'))
                            ->state(function ($record) {
                                $lastRun = $record->getLastRun();
                                return isset($lastRun['duration_seconds'])
                                    ? round($lastRun['duration_seconds'], 2) . ' seconds'
                                    : '-';
                            })
                            ->icon('heroicon-o-clock'),

                        TextEntry::make('employer_id')
                            ->label('Error Message')
                            ->state(fn ($record) => $record->getLastRun()['error_message'] ?? null)
                            ->color('danger')
                            ->visible(fn ($record) => !empty($record->getLastRun()['error_message'] ?? null))
                            ->columnSpanFull()
                            ->copyable(),
                    ])
                    ->visible(fn ($record) => $record->getLastRun() !== null),
                Section::make(__('job::filament.resources.import.form.fields.config'))
                    ->schema([
                        TextEntry::make('config')
                            ->hiddenLabel()
                    ]),
                Section::make(__('job::filament.resources.import.form.sections.run_history'))
                    ->schema([
                        RepeatableEntry::make('runs_history')
                            ->label(__('job::filament.resources.import.form.sections.run_list'))
                            ->state(function ($record) {
                                $runs = $record->runs_history;

                                if (is_string($runs)) {
                                    $runs = json_decode($runs, true);
                                }

                                if (is_array($runs)) {
                                $runs = array_slice(array_reverse($runs), 0, 20);

                                foreach ($runs as $index => &$run) {
                                    $run['_index'] = $index;
                                }

                                return $runs;
                            }

                                return $runs;
                            })
                            ->schema([
                                Section::make()
                                    ->schema([
                                        Grid::make(9)
                                            ->schema([
                                                TextEntry::make('status')
                                                    ->badge()
                                                    ->label(__('job::filament.resources.import.form.fields.status'))
                                                    ->color(fn ($state) => $state === 'success' ? 'success' : 'danger')
                                                    ->formatStateUsing(fn ($state) => ucfirst($state)),

                                                TextEntry::make('started_at')
                                                    ->label(__('job::filament.resources.import.form.fields.started_at'))
                                                    ->dateTime('Y-m-d H:i')
                                                    ->columnSpan(2),

                                                TextEntry::make('duration_seconds')
                                                    ->label(__('job::filament.resources.import.form.fields.duration_seconds'))
                                                    ->formatStateUsing(fn ($state) => round($state, 1) . 's'),

                                                TextEntry::make('fetched_count')
                                                    ->label(__('job::filament.resources.import.form.fields.fetched_count'))
                                                    ->badge()
                                                    ->color('grey'),

                                                TextEntry::make('new_offers')
                                                    ->label(__('job::filament.resources.import.form.fields.new_count'))
                                                    ->color('success')
                                                    ->badge()
                                                    ->formatStateUsing(function ($state) {
                                                        if (is_array($state)) {
                                                            return count($state);
                                                        }
                                                        return 0;
                                                    })
                                                    ->action(
                                                        Actions\Action::make('viewDetails')
                                                            ->icon('heroicon-o-eye')
                                                            ->modalHeading('Nowe oferty')
                                                            ->modalContent(function ($state) {
                                                                $offers = $state ? $state[0] : [];

                                                                if (empty($offers)) {
                                                                    return view('filament::components.empty-state', [
                                                                        'heading' => 'Brak przywróconych ofert',
                                                                        'icon' => 'heroicon-o-inbox',
                                                                    ]);
                                                                }

                                                                return view('filament.modals.offers-table', [
                                                                    'offers' => $offers
                                                                ]);
                                                            })
                                                            ->modalWidth('3xl')
                                                            ->modalSubmitAction(false)
                                                            ->modalCancelActionLabel('Zamknij')
                                                            ->slideOver()
                                                    ),

                                                 TextEntry::make('restored_offers')
                                                    ->label(__('job::filament.resources.import.form.fields.restored_count'))
                                                    ->color('warning')
                                                    ->badge()
                                                    ->formatStateUsing(function ($state) {
                                                        if (is_array($state)) {
                                                            return count($state);
                                                        }
                                                        return 0;
                                                    })
                                                    ->action(
                                                        Actions\Action::make('viewDetails')
                                                            ->icon('heroicon-o-eye')
                                                            ->modalHeading('Przywrócone oferty')
                                                            ->modalContent(function ($state) {
                                                                $offers = $state ? $state[0] : [];
                                                                if (empty($offers)) {
                                                                    return view('filament::components.empty-state', [
                                                                        'heading' => 'Brak przywróconych ofert',
                                                                        'icon' => 'heroicon-o-inbox',
                                                                    ]);
                                                                }

                                                                 return view('filament.modals.offers-table', [
                                                                    'offers' => $offers
                                                                ]);
                                                            })
                                                            ->modalWidth('3xl')
                                                            ->modalSubmitAction(false)
                                                            ->modalCancelActionLabel('Zamknij')
                                                            ->slideOver()
                                                    ),

                                                TextEntry::make('active_offers')
                                                    ->label(__('job::filament.resources.import.form.fields.active_count'))
                                                    ->color('info')
                                                    ->badge()
                                                    ->formatStateUsing(function ($state) {
                                                        if (is_array($state)) {
                                                            return count($state);
                                                        }
                                                        return 0;
                                                    })
                                                    ->action(
                                                        Actions\Action::make('viewDetails')
                                                            ->icon('heroicon-o-eye')
                                                            ->modalHeading('Aktywne oferty')
                                                            ->modalContent(function ($state) {
                                                                $offers = $state ? $state[0] : [];

                                                                if (empty($offers)) {
                                                                    return view('filament::components.empty-state', [
                                                                        'heading' => 'Brak aktywnych ofert',
                                                                        'icon' => 'heroicon-o-inbox',
                                                                    ]);
                                                                }

                                                                 return view('filament.modals.offers-table', [
                                                                    'offers' => $offers
                                                                ]);
                                                            })
                                                            ->modalWidth('3xl')
                                                            ->modalSubmitAction(false)
                                                            ->modalCancelActionLabel('Zamknij')
                                                            ->slideOver()
                                                    ),

                                                TextEntry::make('archived_offers')
                                                    ->label(__('job::filament.resources.import.form.fields.archived_count'))
                                                    ->color('gray')
                                                    ->badge()
                                                    ->formatStateUsing(function ($state) {
                                                        if (is_array($state)) {
                                                            return count($state);
                                                        }
                                                        return 0;
                                                    })
                                                    ->action(
                                                        Actions\Action::make('viewDetails')
                                                            ->icon('heroicon-o-eye')
                                                            ->modalHeading('Zarchiwizowane oferty')
                                                            ->modalContent(function ($state) {
                                                                $offers = $state ? $state[0] : [];

                                                                if (empty($offers)) {
                                                                    return view('filament::components.empty-state', [
                                                                        'heading' => 'Brak zarchiwizowanych ofert',
                                                                        'icon' => 'heroicon-o-inbox',
                                                                    ]);
                                                                }

                                                                 return view('filament.modals.offers-table', [
                                                                    'offers' => $offers
                                                                ]);
                                                            })
                                                            ->modalWidth('3xl')
                                                            ->modalSubmitAction(false)
                                                            ->modalCancelActionLabel('Zamknij')
                                                            ->slideOver()
                                                    ),
                                            ]),
                                        TextEntry::make('error_message')
                                            ->label(__('job::filament.resources.import.form.fields.error_message'))
                                            ->color('danger')
                                            ->visible(fn ($state) => !empty($state))
                                            ->columnSpanFull()
                                            ->copyable(),
                                    ])
                                    ->compact()
                                    ->columnSpanFull(),
                            ])
                            ->columnSpanFull()
                            ->contained(false),
                    ])
                    ->collapsed(true)
                    ->collapsible()
                    ->columnSpanFull(),

            ]);
    }
}
