<?php

namespace MbeGroup\Job\Providers\ImportProviders\Strabag;

use App\ValueObject\Status;
use MbeGroup\Job\Providers\ImportProviders\AbstractProvider;
use Illuminate\Support\Facades\Log;
use Ramsey\Uuid\Uuid;
use Illuminate\Support\Str;

class StrabagProvider extends AbstractProvider
{
    protected const PROVIDER_NAME = 'strabag';
    protected const LIMIT         = 30;

    public function fetchOffers(): array
    {
        $result                = [];
        $availableProviderList = $this->scraperService->getAvailableProviderList(self::PROVIDER_NAME);
        Log::info("Dostępne providery: " . implode(', ', $availableProviderList->all()));
        if ($availableProviderList->isEmpty() || !$availableProviderList->has(self::PROVIDER_NAME)) {
            return $result;
        }

        $jobOfferCount = $this->scraperService->getJobOfferCount(self::PROVIDER_NAME);
        Log::info("Liczba ofert do pobrania: " . $jobOfferCount->get('jobCount'));
        if ($jobOfferCount->isEmpty() || $jobOfferCount->get('jobCount') === 0) {
            return $result;
        }

        $jobOfferList = collect([]);
        $page         = 1;
        $totalPages   = ceil($jobOfferCount->get('jobCount') / self::LIMIT);
        $startMemory  = memory_get_usage();

        for ($page = 1; $page <= $totalPages; $page++) {
            echo "Przetwarzanie strony {$page} / {$totalPages}\n";
            Log::info("Przetwarzanie strony {$page}");

            $pageStartTime = microtime(true);
            $offset        = ($page - 1) * self::LIMIT;

            $maxRetries = 3;
            $attempt    = 0;
            $data       = null;

            while ($attempt < $maxRetries) {
                try {
                    $data = $this->scraperService->getJobOfferList(self::PROVIDER_NAME, $offset, self::LIMIT);
                    break;
                } catch (\Exception $e) {
                    $attempt++;
                    Log::warning("Błąd pobierania strony {$page}, próba {$attempt}/{$maxRetries}: " . $e->getMessage());

                    if ($attempt >= $maxRetries) {
                    Log::error("Nie udało się pobrać strony {$page} po {$maxRetries} próbach. Pomijam.");
                    break;
                    }

                    $sleepSeconds = 5 * $attempt;
                    Log::info("Czekam {$sleepSeconds} sekund przed ponowną próbą...");
                    sleep($sleepSeconds);
                }
            }

            if ($data === null) {
                continue;
            }

            echo "Pobrano " . count($data) . " ofert\n";
            Log::info("Pobrano " . count($data) . " ofert");
            $currentMemoryUsage = memory_get_usage();
            $jobOfferList       = $jobOfferList->merge($data);
            $pageEndTime        = microtime(true);
            $pageDuration       = $pageEndTime - $pageStartTime;
            unset($data);
            echo "Strona {$page} przetworzona w czasie: {$pageDuration} sekund\n";
            echo "Zużycie pamięci: " . ($currentMemoryUsage - $startMemory) . "\n";
            Log::info("Strona {$page} przetworzona w czasie: {$pageDuration} sekund");
            Log::info("Zużycie pamięci: " . ($currentMemoryUsage - $startMemory));

            if ($page < $totalPages) {
            sleep(2);
            }
        };

        Log::info("Pobrano łącznie " . $jobOfferList->count() . " ofert");

        $finalJobOfferList = collect([]);

        $finalJobOfferList = $jobOfferList->flatMap(function ($record) {
            $connectionId = (string) Uuid::uuid1();
            return collect($record['location'])->map(function ($location) use ($record, $connectionId) {
                return [
                    'title'         => $record['title'],
                    'url'           => $record['url'],
                    'location'      => $location,
                    'refNumber'     => $record['refNumber'],
                    'connection_id' => $connectionId,
                    'content'       => $record['content'],
                ];
            });
        })->all();

        Log::info("Łączna liczba ofert po rozbiciu na lokalizacje: " . count($finalJobOfferList));

        $result['offers']                            = $finalJobOfferList;
        $result['finded_offers_count']               = $jobOfferCount->get('jobCount');
        $result['offers_count_after_location_split'] = count($finalJobOfferList);

        return $result;
    }

    protected function transformOffer(array $raw): array
    {
        return [
            // === WYMAGANE POLA ===

            // Unique provider ID
            'provider_id' => $this->makeProviderId(
                $raw['refNumber'],
                $raw['location']
            ),

            // Basic info
            'title'            => $raw['title'],
            'slug'             => Str::slug($raw['title'] . ' ' . $raw['refNumber'] . ' ' . $raw['location']),
            'reference_number' => $raw['refNumber'],
            'status'           => Status::IN_PROGRESS,

            // Connection ID - łączy oferty z różnymi lokalizacjami
            'connection_id' => $raw['connection_id'],

            // Locations - WYMAGANE (co najmniej jedna)
            'locations' => [$this->searchPlaceByName($raw['location'])],

            'industry_ids'       => null,
            'specialization_ids' => null,

            // === CATEGORY - OPCJONALNE ===
            'position_id'           => null,
            'position_level_id'     => null,
            'is_for_technicians'    => false,
            'operating_mode_ids'    => null,
            'is_remote_recruitment' => false,
            'hourly_rate_id'        => null,

            // === REQUIREMENTS - arrays ===
            'preferred_study_ids' => [],

            'workplace_type_ids'   => [],
            'work_system_id'       => null,
            'delegations_id'       => null,
            'contract_duration_id' => null,
            'salary_structure_ids' => null,
            'has_team_management'  => false,
            'has_driving_license'  => false,
             'no_experience'   => false,
             'has_company_car' => false,

            // === NICE TO HAVE - arrays ===
            'nice_to_have_language_ids'                    => [],
            'nice_to_have_technologies_and_tools_ids'      => [],
            'nice_to_have_skills_and_methodologies_ids'    => [],
            'nice_to_have_licenses_and_qualifications_ids' => [],

            // === APPLICATION METHOD - WYMAGANE ===
            'application_method' => [
                'apply_options' => [
                    'application_method'               => 'employer-link',
                    'application_link_reference'       => $raw['url'],
                    'application_link_reference_value' => null,
                ]
            ],

            // === SALARY - OPCJONALNE (ale może być wymagane w formularzu) ===
            'salary'            => [],
            'salary_for_search' => null,

            // === TEMPLATE ===
            'job_offer_template_name' => null,
            'template_variables'      => [
                'content' => $raw['content']
            ],
            'template' => null,

            // === PUBLICATION SETTINGS ===
            'publication_date'  => now(),
            'expiration_date'   => now()->addMonths(1), // Domyślnie 1 miesiąc
            'is_top_offer'      => false,
            'refresh_frequency' => null,

            // === PRODUCT ===
            'product_statistics' => null,

            // === META ===
            'created_by' => 'import_strabag',
            'updated_by' => 'import_strabag',
            'created_at' => now(),
            'updated_at' => now(),
        ];
    }

    /**
     * Map raw offer to JobOffer structure
     */
    protected function mapOffer(array $raw, ?string $location): array
    {
        $referenceNumber = $raw['reference_number'] ?? $raw['id'] ?? uniqid();

        return [
            'provider_id'      => $this->makeProviderId($referenceNumber, $location ?? 'remote'),
            'title'            => $raw['title'] ?? $raw['job_title'] ?? 'Untitled',
            'reference_number' => $referenceNumber,
            'status'           => 'pending',                                                      // Nowe oferty zawsze jako pending
            'locations'        => $location ? [$this->formatLocation($location)] : [],

            // Opcjonalne pola - dostosuj do API
            'industry_ids'       => $raw['industry_ids'] ?? null,
            'specialization_ids' => $raw['specialization_ids'] ?? null,
            'position_id'        => $raw['position_id'] ?? null,
            'position_level_id'  => $raw['position_level_id'] ?? null,
            'operating_mode_ids' => $raw['operating_mode_ids'] ?? null,
            'employment_type_id' => $raw['employment_type_id'] ?? null,

            'template_variables' => $this->extractTemplateVariables($raw),
            'application_method' => $raw['application_url'] ?? $raw['apply_link'] ?? null,

            // Timestamps
            'publication_date' => isset($raw['published_at']) ? new \DateTime($raw['published_at']) : now(),
            'expiration_date'  => isset($raw['expires_at']) ? new \DateTime($raw['expires_at']) : null,
        ];
    }

    /**
     * Format location to JobOffer location structure
     */
    protected function formatLocation(string $location): array
    {
        return [
            'location_name'       => $location,
            'municipality'        => null,
            'region'              => null,
            'postal_code'         => null,
            'status'              => 'active',
            'location_is_default' => true,
            'latitude'            => null,
            'longitude'           => null,
        ];
    }

    /**
     * Extract template variables from raw data
     */
    protected function extractTemplateVariables(array $raw): array
    {
        return [
            'description'      => $raw['description'] ?? '',
            'requirements'     => $raw['requirements'] ?? '',
            'responsibilities' => $raw['responsibilities'] ?? '',
            'benefits'         => $raw['benefits'] ?? [],
            'salary_info'      => $raw['salary'] ?? null,
        ];
    }
}
