<?php

namespace MbeGroup\Job\Services;

use MbeGroup\Job\Contracts\JobOffersSearchServiceInterface;
use Elastic\Elasticsearch\Client;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Pagination\LengthAwarePaginator as ConcreteLengthAwarePaginator;
use App\Support\FilterNormalizer;

class JobOffersSearchService implements JobOffersSearchServiceInterface
{
    protected Client $client;
    protected string $index = 'job_offers';

    public function __construct(Client $client)
    {
        $this->client = $client;
    }

    public function search(array $filters, int $perPage, int $page): LengthAwarePaginator
    {
        $esQuery = $this->buildElasticsearchQuery($filters, $perPage, $page);

        $response = $this->client->search($esQuery);
        $results  = $response->asArray();

        $hits  = $results['hits']['hits'] ?? [];
        $total = $results['hits']['total']['value'] ?? 0;

        $processedData = $this->processResults($hits);
        return new ConcreteLengthAwarePaginator(
            $processedData,
            $total,
            $perPage,
            $page,
            [
                'path'  => request()->url(),
                'query' => request()->query(),
            ]
        );
    }

    private function buildElasticsearchQuery(array $filters, int $perPage, int $page): array
    {
        $query = [
            'index' => $this->index,
            'body'  => [
                'from'  => ($page - 1) * $perPage,
                'size'  => $perPage,
                'query' => [
                    'bool' => [
                        'must'     => [],
                        'filter'   => [],
                        'must_not' => [],
                    ],
                ],
                'sort' => $this->buildSort($filters),
            ],
        ];

        if (!empty($filters['q'])) {
            $query['body']['query']['bool']['must'][] = [
                'multi_match' => [
                    'query'  => $filters['q'],
                    'fields' => ['title^3', 'description', 'requirements'],
                ],
            ];
        }

        if ($filters['industry_ids'] ?? null) {
            $industryIds                                = FilterNormalizer::toIntArray($filters['industry_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['industry_ids' => $industryIds]
            ];
        }

        if ($filters['specialization_ids'] ?? null) {
            $specializationIds                          = FilterNormalizer::toIntArray($filters['specialization_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['specialization_ids' => $specializationIds]
            ];
        }

        if ($filters['employer_ids'] ?? null) {
            $employerIds                                = FilterNormalizer::toIntArray($filters['employer_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['employer.id' => $employerIds]
            ];
        }

        if (!empty($filters['lat']) && !empty($filters['lon'])) {
            $lat      = (float) $filters['lat'];
            $lon      = (float) $filters['lon'];
            $distance = (int) ($filters['distance'] ?? 50);

            $query['body']['query']['bool']['filter'][] = [
                'nested' => [
                    'path'  => 'locations',
                    'query' => [
                        'geo_distance' => [
                            'distance'           => "{$distance}km",
                            'locations.location' => [
                                'lat' => $lat,
                                'lon' => $lon
                            ]
                        ]
                    ]
                ]
            ];
        }

        if ($filters['position_ids'] ?? null) {
            $positionIds                                = FilterNormalizer::toIntArray($filters['position_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['position_id' => $positionIds]
            ];
        }

        if ($filters['position_level_ids'] ?? null) {
            $positionLevelIds                           = FilterNormalizer::toIntArray($filters['position_level_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['position_level_id' => $positionLevelIds]
            ];
        }

        if ($filters['operating_mode_ids'] ?? null) {
            $operatingModeIds                           = FilterNormalizer::toIntArray($filters['operating_mode_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['operating_mode_ids' => $operatingModeIds]
            ];
        }

        if ($filters['salary_employment_type_ids'] ?? null) {
            $employmentTypeIds                          = FilterNormalizer::toIntArray($filters['salary_employment_type_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['employment_type_id' => $employmentTypeIds]
            ];
        }

        if ($filters['hourly_rate_ids'] ?? null) {
            $hourlyRateIds                              = FilterNormalizer::toIntArray($filters['hourly_rate_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['hourly_rate_id' => $hourlyRateIds]
            ];
        }

        if (!empty($filters['only_with_salary']) && $filters['only_with_salary']) {
            $query['body']['query']['bool']['filter'][] = [
                'exists' => ['field' => 'salary_for_search']
            ];
        }

        if (!empty($filters['min_salary'])) {
            $minSalary                                  = (int) $filters['min_salary'];
            $query['body']['query']['bool']['filter'][] = [
                'range' => [
                    'salary_for_search.to' => [
                    'gte' => $minSalary
                    ]
                ]
            ];
        }

        if ($filters['preferred_study_ids'] ?? null) {
            $preferredStudyIds                          = FilterNormalizer::toIntArray($filters['preferred_study_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['preferred_study_ids' => $preferredStudyIds]
            ];
        }

        if ($filters['workplace_type_ids'] ?? null) {
            $workplaceTypeIds                          = FilterNormalizer::toIntArray($filters['workplace_type_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['workplace_type_ids' => $workplaceTypeIds]
            ];
        }

        if (!empty($filters['work_system_id'])) {
            $workSystemId                               = (int) $filters['work_system_id'];
            $query['body']['query']['bool']['filter'][] = [
                'term' => ['work_system_id' => $workSystemId]
            ];
        }

        if (!empty($filters['delegations_id'])) {
            $delegationsId                               = (int) $filters['delegations_id'];
            $query['body']['query']['bool']['filter'][] = [
                'term' => ['delegations_id' => $delegationsId]
            ];
        }

        if (!empty($filters['contract_duration_id'])) {
            $contractDurationId                               = (int) $filters['contract_duration_id'];
            $query['body']['query']['bool']['filter'][] = [
                'term' => ['contract_duration_id' => $contractDurationId]
            ];
        }

        if ($filters['salary_structure_ids'] ?? null) {
            $salaryStructureIds                          = FilterNormalizer::toIntArray($filters['salary_structure_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['salary_structure_ids' => $salaryStructureIds]
            ];
        }

        if ($filters['has_team_management'] ?? null) {
            $hasTeamManagement                          = (bool) $filters['has_team_management'];
            $query['body']['query']['bool']['filter'][] = [
                'term' => ['has_team_management' => $hasTeamManagement]
            ];
        }

        if ($filters['has_driving_license'] ?? null) {
            $hasDrivingLicense                          = (bool) $filters['has_driving_license'];
            $query['body']['query']['bool']['filter'][] = [
                'term' => ['has_driving_license' => $hasDrivingLicense]
            ];
        }

        if ($filters['no_experience'] ?? null) {
            $noExperience                          = (bool) $filters['no_experience'];
            $query['body']['query']['bool']['filter'][] = [
                'term' => ['no_experience' => $noExperience]
            ];
        }

        if ($filters['has_company_car'] ?? null) {
            $hasCompanyCar                          = (bool) $filters['has_company_car'];
            $query['body']['query']['bool']['filter'][] = [
                'term' => ['has_company_car' => $hasCompanyCar]
            ];
        }

        if (!empty($filters['is_remote_recruitment'])) {
            $isRemote                                   = (bool) $filters['is_remote_recruitment'];
            $query['body']['query']['bool']['filter'][] = [
                'term' => ['is_remote_recruitment' => $isRemote]
            ];
        }

        if (!empty($filters['is_for_technicians'])) {
            $isForTechnicians                           = (bool) $filters['is_for_technicians'];
            $query['body']['query']['bool']['filter'][] = [
                'term' => ['is_for_technicians' => $isForTechnicians]
            ];
        }

        if ($filters['company_size_ids'] ?? null) {
            $companySizeIds                             = FilterNormalizer::toIntArray($filters['company_size_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['employer.company_size_id' => $companySizeIds]
            ];
        }

        if ($filters['company_type_ids'] ?? null) {
            $companyTypeIds                             = FilterNormalizer::toIntArray($filters['company_type_ids']);
            $query['body']['query']['bool']['filter'][] = [
                'terms' => ['employer.company_type_id' => $companyTypeIds]
            ];
        }

        if (!empty($filters['exclude_ids'])) {
            $query['body']['query']['bool']['must_not'][] = [
                'ids' => ['values' => $filters['exclude_ids']]
            ];
        }

        if (!empty($filters['publication_date_from'])) {
            $query['body']['query']['bool']['filter'][] = [
                'range' => [
                    'publication_date' => [
                        'gte' => strtotime($filters['publication_date_from'])
                    ]
                ]
            ];
        }

        return $query;
    }

    private function processResults(array $hits): array
    {
        $results = [];
        foreach ($hits as $hit) {
            $results[] = $hit['_source'];
        }
        return $results;
    }

    private function buildSort(array $filters): array
    {
        $sortBy = $filters['sort_by'] ?? 'newest';

        return match ($sortBy) {
            'oldest' => [
                ['created_at' => ['order' => 'asc']]
            ],
            'newest' => [
                ['created_at' => ['order' => 'desc']]
            ],
            default => [
                ['created_at' => ['order' => 'desc']]
            ]
        };
    }
}
