<?php

namespace MbeGroup\Job\Providers\ImportProviders;

use MbeGroup\Job\Models\Import;
use InvalidArgumentException;
use MbeGroup\Job\Providers\ImportProviders\AbstractProvider;
use Illuminate\Support\Str;
use MbeGroup\Job\Contracts\ScraperJobOfferServiceInterface;

class ProviderFactory
{
        /**
     * Create provider instance from import
     */
    public function make(Import $import): AbstractProvider
    {
        $providerClass = $this->resolveProviderClass($import->provider_type);
        $scraperService = app()->make(ScraperJobOfferServiceInterface::class);

        if (!class_exists($providerClass)) {
            throw new InvalidArgumentException(
                "Provider class '{$providerClass}' does not exist"
            );
        }

        if (!is_subclass_of($providerClass, AbstractProvider::class)) {
            throw new InvalidArgumentException(
                "Provider class '{$providerClass}' must extend AbstractProvider"
            );
        }


        return new $providerClass($import, $scraperService);
    }

    /**
     * Resolve provider class name from provider type
     * Example: 'axa' -> App\Models\Providers\ImportProviders\Axa\AxaProvider
     */
    protected function resolveProviderClass(string $providerType): string
    {
        $className = Str::studly($providerType) . 'Provider';
        $namespace = 'MbeGroup\\Job\\Providers\\ImportProviders\\' . Str::studly($providerType);

        return $namespace . '\\' . $className;
    }

    /**
     * Get available provider types with labels
     */
    public static function getAvailableProviders(): array
    {
        return [
            'axa' => 'AXA Careers',
        ];
    }

    /**
     * Check if provider type exists
     */
    public static function exists(string $providerType): bool
    {
        return array_key_exists($providerType, self::getAvailableProviders());
    }

    /**
     * Get provider type label
     */
    public static function getLabel(string $providerType): ?string
    {
        return self::getAvailableProviders()[$providerType] ?? null;
    }
}
