<?php

namespace MbeGroup\Job\Controllers\Api\V1;

use App\Http\Controllers\Api\V1\CrudApiController;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use MbeGroup\Job\Resources\FavoriteResource;
use MbeGroup\Job\Contracts\FavoriteServiceInterface;
use MbeGroup\Job\Models\Favorite;
use MbeGroup\Job\Requests\FavoriteRequest;

final class FavoriteController extends CrudApiController
{
    protected string $resourceName = 'favorite';

    public function __construct(
        protected FavoriteServiceInterface $favoriteService,
    ) {
    }

    public function index(Request $request): JsonResponse
    {
        $user      = auth()->user();
        $limit     = (int) $request->input('limit', 15);
        $page      = (int) $request->input('page', 1);
        $favorites = $this->favoriteService->getFavoritesForUser($user, $page, $limit);

        return $this->success(FavoriteResource::collection($favorites), [
            'pagination' => [
                'total'        => $favorites->total(),
                'per_page'     => $favorites->perPage(),
                'current_page' => $favorites->currentPage(),
                'last_page'    => $favorites->lastPage(),
            ],
        ]);
    }

    public function show(string $id): JsonResponse
    {
        return $this->notImplemented('show');
    }

    public function store(Request $request): JsonResponse
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error('Użytkownik nie jest zalogowany.', 401);
        }

        $formRequest = app(FavoriteRequest::class);
        $validated   = $formRequest->validated();
        $jobOffer    = $formRequest->getJobOffer();

        try {
            $favorite = Favorite::create([
                'job_offer_id' => $jobOffer->_id,
                'user_id'      => $user->id,
            ]);

            return $this->success(
                new FavoriteResource($favorite),
                [],
                'Ulubiona oferta została pomyślnie dodana.'
            );
        } catch (\Exception $e) {
            \Log::error('Błąd podczas dodawania ulubionej oferty: ' . $e->getMessage());
            return $this->error('Nie można dodać ulubionej oferty. Proszę spróbować ponownie później.', 500);
        }
    }

    public function update(Request $request, string $id): JsonResponse
    {
        return $this->notImplemented('update');
    }

    public function destroy(string $id): JsonResponse
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error('Użytkownik nie jest zalogowany.', 401);
        }

        try {
            $favorite = Favorite::where('user_id', $user->id)
                ->where('job_offer_id', $id)
                ->first();

            if (!$favorite) {
                return $this->error('Ulubiona oferta nie została znaleziona.', 404);
            }

            $favorite->delete();

            return $this->success(null, [], 'Ulubiona oferta została pomyślnie usunięta.');
        } catch (\Exception $e) {
            \Log::error('Błąd podczas usuwania ulubionej oferty: ' . $e->getMessage());
            return $this->error('Nie można usunąć ulubionej oferty. Proszę spróbować ponownie później.', 500);
        }
    }
}
