<?php

namespace MbeGroup\Job\Controllers\Api\V1;

use App\Http\Controllers\Api\V1\CrudApiController;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use MbeGroup\Job\Contracts\JobOfferRecommendationServiceInterface;
use MbeGroup\Job\Models\Recommendation;
use MbeGroup\Job\Requests\StoreRecommendationRequest;
use MbeGroup\Job\Resources\RecommendationResource;

final class RecommendationController extends CrudApiController
{
    protected string $resourceName = 'recommendation';

    public function __construct(
        protected JobOfferRecommendationServiceInterface $recommendationService,
    ) {
    }

    public function index(Request $request): JsonResponse
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error('Użytkownik nie jest zalogowany.', [], 401);
        }
        try {
            $recommendations = $this->recommendationService->findUserJobRecomendation($user->id);
            return $this->success(RecommendationResource::collection($recommendations));
        } catch (\Exception $e) {
            \Log::error('Błąd podczas pobierania rekomendacji: ' . $e->getMessage());
            return $this->error('Nie można pobrać rekomendacji. Proszę spróbować ponownie później.', [], 500);
        }
    }

    public function show(string $id): JsonResponse
    {
        return $this->notImplemented('show');
    }

    public function store(Request $request): JsonResponse
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error('Użytkownik nie jest zalogowany.', [], 401);
        }
        try {
            $formRequest    = app(StoreRecommendationRequest::class);
            $validated      = $formRequest->validated();
            $data           = $this->recommendationService->prepareDataFromRequest($validated);
            $recommendation = $this->recommendationService->addFilterDataToUserRecommendation($data, $validated['type']);

                return $this->success(
                    new RecommendationResource($recommendation),
                    [],
                    'Rekomendacja została pomyślnie dodana.'
                );
        } catch (\Exception $e) {
            \Log::error('Błąd podczas dodawania rekomendacji: ' . $e->getMessage());
            return $this->error('Nie można dodać rekomendacji. Proszę spróbować ponownie później.', [], 500);
        }
    }

    public function update(Request $request, string $id): JsonResponse
    {
        return $this->notImplemented('update');
    }

    public function destroy(string $id): JsonResponse
    {
        return $this->notImplemented('destroy');
    }

    public function destroyAllForUser(): JsonResponse
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error('Użytkownik nie jest zalogowany.', 401);
        }
        try {
            Recommendation::where('user_id', $user->id)->delete();
            return $this->success(null, [], 'Wszystkie rekomendacje zostały pomyślnie usunięte.');
        } catch (\Exception $e) {
            \Log::error('Błąd podczas usuwania rekomendacji: ' . $e->getMessage());
            return $this->error('Nie można usunąć rekomendacji. Proszę spróbować ponownie później.', [], 500);
        }
    }

    public function getRecommendationByType(Request $request, string $type): JsonResponse
    {
        $user = auth()->user();
        if (!$user) {
            return $this->error('Użytkownik nie jest zalogowany.', 401);
        }
        try {
            $excludedFilters = $request->excluded_filters ?? [];
            $recommendation = $this->recommendationService->findUserJobRecomendationByType($user->id, $type, $excludedFilters);

            return $this->success($recommendation);
        } catch (\Exception $e) {
            \Log::error('Błąd podczas pobierania rekomendacji: ' . $e->getMessage());
            return $this->error('Nie można pobrać rekomendacji. Proszę spróbować ponownie później.', [], 500);
        }
    }
}
