<?php

namespace MbeGroup\Job\Filament\Resources\JobOfferResource\Pages;

use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use MbeGroup\Job\Filament\Resources\JobOfferResource;
use MbeGroup\Job\Contracts\JobOfferContentServiceInterface;
use MbeGroup\Job\Contracts\JobOffersSalaryServiceInterface;

class EditJobOffer extends EditRecord
{
    protected static string $resource = JobOfferResource::class;
    protected function getHeaderActions(): array
    {
        return [
            Actions\DeleteAction::make(),
        ];
    }

    protected function getFormActions(): array
    {
        return [
            $this->getSaveFormAction(),
            Actions\Action::make('saveAndBack')
                ->label(__('filament.general.saveAndBack'))
                ->color('gray')
                ->action(function () {
                    $this->save();
                    $this->redirect($this->getResource()::getUrl('index'));
                })
                ->keyBindings(['mod+shift+s']),
            $this->getCancelFormAction(),
        ];
    }

    protected function mutateFormDataBeforeSave(array $data): array
    {
        $user                                = auth()->user();
        $template                            = app(JobOfferContentServiceInterface::class)->getTemplateByEmployerIdAndTemplateName($data['employer_id'], $data['job_offer_template_name']);
        $data['updated_by']                  = $user ? $user->name . ' '. $user->surname : 'system';
        $data['template']                    = app(JobOfferContentServiceInterface::class)->generateContent($data);
        $data['template_css']                = $template['job_offer_template_css'] ?? null;
        $data['salary_for_search']           = app(JobOffersSalaryServiceInterface::class)->getSearchSalaryRange($data['salary'] ?? []);
        $data['template_benefits']           = isset($template['job_offer_template_benefits']) ? $template['job_offer_template_benefits'] : [];
        $data['template_recruitment_stages'] = isset($template['job_offer_template_recruitment_stages']) ? $template['job_offer_template_recruitment_stages'] : [];
        $data['template_about_project']      = isset($template['job_offer_template_about_project']) ? $template['job_offer_template_about_project'] : null;

        return $data;
    }

    protected function afterSave(): void
    {
        $this->syncProductStatistics();
    }

    protected function syncProductStatistics(): void
    {
        $record = $this->getRecord();
        if ($record->wasChanged('product_statistics')) {
            try {
                $oldProductStatisticsId = $record->getPrevious()['product_statistics'];
                $newProductStatisticsId = $record->product_statistics;
                $employer               = $record->employer;

                if ($newProductStatisticsId) {
                    $newProductStatistics = \MbeGroup\Product\Models\ProductStatistics::find($newProductStatisticsId);
                    $oldProductStatistics = \MbeGroup\Product\Models\ProductStatistics::find($oldProductStatisticsId);
                    $products             = $employer->products;
                    $isNoLimit            = $this->isNoLimitProduct($newProductStatistics, $products);
                    $isOldSlot            = $oldProductStatistics ? $this->isSlotProduct($oldProductStatistics, $products) : false;

                    if (!$isNoLimit && $newProductStatistics && $newProductStatistics->quantity > 0) {
                        $isSlot = $this->isSlotProduct($newProductStatistics, $products);
                        $newProductStatistics->decrement('quantity');
                        $index = array_search($newProductStatistics->employer_item_id, array_column($products, 'item_id'));

                        if ($index !== false) {
                            $products[$index]['quantity'] -= 1;
                            if (!$isSlot && $products[$index]['quantity'] <= 0) {
                                unset($products[$index]);
                                $products = array_values($products);
                            }

                            $employer->products = $products;
                            $employer->save();
                        }
                    }

                    if ($isOldSlot) {
                        $oldProductStatistics->increment('quantity');
                        $oldIndex = array_search($oldProductStatistics->employer_item_id, array_column($products, 'item_id'));

                        if ($oldIndex !== false) {
                            $products[$oldIndex]['quantity'] += 1;
                            $employer->products               = $products;
                            $employer->save();
                        }
                    }

                }
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::error('Error syncing product statistics: ' . $e->getMessage());
            } catch (\Throwable $t) {
                \Illuminate\Support\Facades\Log::error('Error syncing product statistics: ' . $t->getMessage());
            }
        }
    }

    private function isNoLimitProduct($productStatistics, $products): bool
    {
        $newProductId    = $productStatistics->product_id;
        $selectedProduct = collect($products)->firstWhere('product_id', $newProductId);

        if ($selectedProduct && isset($selectedProduct['is_no_limit']) && $selectedProduct['is_no_limit']) {
            return true;
        }

        return false;
    }

    private function isSlotProduct($productStatistics, $products): bool
    {
        $newProductId    = $productStatistics->product_id;
        $selectedProduct = collect($products)->firstWhere('product_id', $newProductId);

        if ($selectedProduct && isset($selectedProduct['is_slot']) && $selectedProduct['is_slot']) {
            return true;
        }

        return false;
    }
}
